/** @file
 * @brief A termlist containing all terms in a glass database.
 */
/* Copyright (C) 2005,2007,2008,2009,2010,2011,2024 Olly Betts
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#ifndef XAPIAN_INCLUDED_GLASS_ALLTERMSLIST_H
#define XAPIAN_INCLUDED_GLASS_ALLTERMSLIST_H

#include "backends/alltermslist.h"
#include "glass_database.h"
#include "glass_postlist.h"

#include <string_view>

class GlassCursor;

class GlassAllTermsList : public AllTermsList {
    /// Copying is not allowed.
    GlassAllTermsList(const GlassAllTermsList &);

    /// Assignment is not allowed.
    void operator=(const GlassAllTermsList &);

    /// Keep a reference to our database to stop it being deleted.
    Xapian::Internal::intrusive_ptr<const GlassDatabase> database;

    /** A cursor which runs through the postlist table reading termnames from
     *  the keys.
     */
    GlassCursor * cursor;

    /// The prefix to restrict the terms to.
    std::string prefix;

    /** The term frequency of the term at the current position.
     *
     *  If this value is zero, then we haven't read the term frequency for the
     *  current term yet.  We need to call read_termfreq() to read this.
     */
    mutable Xapian::doccount termfreq;

    /// Read and cache the term frequency.
    void read_termfreq() const;

  public:
    GlassAllTermsList(Xapian::Internal::intrusive_ptr<const GlassDatabase> database_,
		      std::string_view prefix_)
	: database(database_), cursor(NULL), prefix(prefix_), termfreq(0) { }

    /// Destructor.
    ~GlassAllTermsList();

    Xapian::termcount get_approx_size() const;

    /** Returns the term frequency of the current term.
     *
     *  Either next() or skip_to() must have been called before this
     *  method can be called.
     */
    Xapian::doccount get_termfreq() const;

    /// Advance to the next term in the list.
    TermList * next();

    /// Advance to the first term which is >= tname.
    TermList* skip_to(std::string_view tname);
};

#endif /* XAPIAN_INCLUDED_GLASS_ALLTERMSLIST_H */
