/** @file
 * @brief Postlists in glass databases
 */
/* Copyright 1999,2000,2001 BrightStation PLC
 * Copyright 2002 Ananova Ltd
 * Copyright 2002-2024 Olly Betts
 * Copyright 2007,2009 Lemur Consulting Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#ifndef XAPIAN_INCLUDED_GLASS_POSTLIST_H
#define XAPIAN_INCLUDED_GLASS_POSTLIST_H

#include <xapian/database.h>

#include "backends/leafpostlist.h"
#include "glass_defs.h"
#include "glass_inverter.h"
#include "glass_positionlist.h"
#include "omassert.h"

#include <memory>
#include <map>
#include <string>
#include <string_view>

class GlassCursor;
class GlassDatabase;

namespace Glass {
    class PostlistChunkReader;
    class PostlistChunkWriter;
    class RootInfo;
}

using Glass::RootInfo;

/** A postlist in a glass database.
 */
class GlassPostList : public LeafPostList {
    /** The database we are searching.  This pointer is held so that the
     *  database doesn't get deleted before us, and also to give us access
     *  to the position_table.
     */
    Xapian::Internal::intrusive_ptr<const GlassDatabase> this_db;

    /// The position list object for this posting list.
    GlassRePositionList* positionlist = NULL;

    /// Whether we've started reading the list yet.
    bool have_started;

    /// True if this is the last chunk.
    bool is_last_chunk;

    /// Whether we've run off the end of the list yet.
    bool is_at_end;

    /// Cursor pointing to current chunk of postlist.
    std::unique_ptr<GlassCursor> cursor;

    /// The first document id in this chunk.
    Xapian::docid first_did_in_chunk;

    /// The last document id in this chunk.
    Xapian::docid last_did_in_chunk;

    /// Position of iteration through current chunk.
    const char * pos;

    /// Pointer to byte after end of current chunk.
    const char * end;

    /// Document id we're currently at.
    Xapian::docid did;

    /// The wdf of the current document.
    Xapian::termcount wdf;

    /// Upper bound on wdf for this postlist.
    Xapian::termcount wdf_upper_bound;

    /// Copying is not allowed.
    GlassPostList(const GlassPostList &);

    /// Assignment is not allowed.
    void operator=(const GlassPostList &);

    /** Move to the next item in the chunk, if possible.
     *  If already at the end of the chunk, returns false.
     */
    bool next_in_chunk();

    /** Move to the next chunk.
     *
     *  If there are no more chunks in this postlist, this will set
     *  is_at_end to true.
     */
    void next_chunk();

    /** Return true if the given document ID lies in the range covered
     *  by the current chunk.  This does not say whether the document ID
     *  is actually present.  It will return false if the document ID
     *  is greater than the last document ID in the chunk, even if it is
     *  less than the first document ID in the next chunk: it is possible
     *  for no chunk to contain a particular document ID.
     */
    bool current_chunk_contains(Xapian::docid desired_did);

    /** Move to chunk containing the specified document ID.
     *
     *  This moves to the chunk whose starting document ID is
     *  <= desired_did, but such that the next chunk's starting
     *  document ID is > desired_did.
     *
     *  It is thus possible that current_chunk_contains(desired_did)
     *  will return false after this call, since the document ID
     *  might lie after the end of this chunk, but before the start
     *  of the next chunk.
     */
    void move_to_chunk_containing(Xapian::docid desired_did);

    /** Scan forward in the current chunk for the specified document ID.
     *
     *  This is particularly efficient if the desired document ID is
     *  greater than the last in the chunk - it then skips straight
     *  to the end.
     *
     *  @return true if we moved to a valid document,
     *	    false if we reached the end of the chunk.
     */
    bool move_forward_in_chunk_to_at_least(Xapian::docid desired_did);

    GlassPostList(Xapian::Internal::intrusive_ptr<const GlassDatabase> this_db_,
		  std::string_view term,
		  GlassCursor * cursor_);

    void init();

  public:
    /// Default constructor.
    GlassPostList(Xapian::Internal::intrusive_ptr<const GlassDatabase> this_db_,
		  std::string_view term,
		  bool keep_reference);

    /// Destructor.
    ~GlassPostList();

    bool open_nearby_postlist(std::string_view term_,
			      bool need_read_pos,
			      LeafPostList*& pl) const;

    /** Used for looking up doclens.
     *
     *  @return true if docid @a desired_did has a document length.
     */
    bool jump_to(Xapian::docid desired_did);

    /// Returns the current docid.
    Xapian::docid get_docid() const { Assert(have_started); return did; }

    /** Returns the Within Document Frequency of the term in the current
     *  document.
     */
    Xapian::termcount get_wdf() const { Assert(have_started); return wdf; }

    /** Get the list of positions of the term in the current document.
     */
    PositionList *read_position_list();

    /** Get the list of positions of the term in the current document.
     */
    PositionList * open_position_list() const;

    /// Move to the next document.
    PostList * next(double w_min);

    /// Skip to next document with docid >= docid.
    PostList * skip_to(Xapian::docid desired_did, double w_min);

    /// Return true if and only if we're off the end of the list.
    bool at_end() const { return is_at_end; }

    Xapian::termcount get_wdf_upper_bound() const;

    void get_docid_range(Xapian::docid& first, Xapian::docid& last) const;

    /// Get a description of the document.
    std::string get_description() const;

    /// Read the term frequency and collection frequency.
    static void read_freqs(const char** posptr,
			   const char* end,
			   Xapian::doccount* number_of_entries_ptr,
			   Xapian::termcount* collection_freq_ptr);
};

class GlassPostListTable : public GlassTable {
    /// PostList for looking up document lengths.
    mutable std::unique_ptr<GlassPostList> doclen_pl;

  public:
    /** Create a new table object.
     *
     *  This does not create the table on disk - the create() method must
     *  be called before the table is created on disk
     *
     *  This also does not open the table - the open() method must be
     *  called before use is made of the table.
     *
     *  @param path_          - Path at which the table is stored.
     *  @param readonly_      - whether to open the table for read only
     *                          access.
     */
    GlassPostListTable(const std::string& path_, bool readonly_)
	: GlassTable("postlist", path_ + "/postlist.", readonly_),
	  doclen_pl()
    { }

    GlassPostListTable(int fd, off_t offset_, bool readonly_)
	: GlassTable("postlist", fd, offset_, readonly_),
	  doclen_pl()
    { }

    void open(int flags_, const RootInfo & root_info,
	      glass_revision_number_t rev) {
	doclen_pl.reset(0);
	GlassTable::open(flags_, root_info, rev);
    }

    /// Merge changes for a term.
    void merge_changes(std::string_view term,
		       const Inverter::PostingChanges& changes);

    /// Merge document length changes.
    void merge_doclen_changes(const std::map<Xapian::docid,
					Xapian::termcount>& doclens);

    Xapian::docid get_chunk(std::string_view tname,
			    Xapian::docid did, bool adding,
			    Glass::PostlistChunkReader** from,
			    Glass::PostlistChunkWriter** to);

    /// Compose a key from a termname and docid.
    static std::string make_key(std::string_view term, Xapian::docid did) {
	return pack_glass_postlist_key(term, did);
    }

    /// Compose a key from a termname.
    static std::string make_key(std::string_view term) {
	return pack_glass_postlist_key(term);
    }

    bool term_exists(std::string_view term) const {
	return key_exists(make_key(term));
    }

    /** Returns frequencies for a term.
     *
     *  @param term		The term to get frequencies for
     *  @param termfreq_ptr	Point to return number of docs indexed by @a
     *			term (or NULL not to return)
     *  @param collfreq_ptr	Point to return number of occurrences of @a
     *			term in the database (or NULL not to return)
     *  @param wdfub_ptr	Point to return an upper bound on the wdf
     *			of @a term in the database (or NULL not to
     *			return)
     */
    void get_freqs(std::string_view term,
		   Xapian::doccount* termfreq_ptr,
		   Xapian::termcount* collfreq_ptr,
		   Xapian::termcount* wdfub_ptr = NULL) const;

    /** Returns the length of document @a did. */
    Xapian::termcount get_doclength(Xapian::docid did,
				    Xapian::Internal::intrusive_ptr<const GlassDatabase> db) const;

    /** Check if document @a did exists. */
    bool document_exists(Xapian::docid did,
			 Xapian::Internal::intrusive_ptr<const GlassDatabase> db) const;

    void get_used_docid_range(Xapian::docid & first,
			      Xapian::docid & last) const;
};
#ifdef DISABLE_GPL_LIBXAPIAN
# error GPL source we cannot relicense included in libxapian
#endif

#endif /* XAPIAN_INCLUDED_GLASS_POSTLIST_H */
