/*
** 2000-05-29
**
** The author disclaims copyright to this source code.  In place of
** a legal notice, here is a blessing:
**
**    May you do good and not evil.
**    May you find forgiveness for yourself and forgive others.
**    May you share freely, never taking more than you give.
**
*************************************************************************
** Driver template for the LEMON parser generator.
**
** Synced with upstream:
** https://www.sqlite.org/src/artifact/468a155e8729cfbccfe1d85bf60d064f1dab76167a51149ec5c7928a2de63953
**
** The "lemon" program processes an LALR(1) input grammar file, then uses
** this template to construct a parser.  The "lemon" program inserts text
** at each "%%" line.  Also, any "P-a-r-s-e" identifier prefix (without the
** interstitial "-" characters) contained in this template is changed into
** the value of the %name directive from the grammar.  Otherwise, the content
** of this template is copied straight through into the generate parser
** source file.
**
** The following is the concatenation of all %include directives from the
** input grammar file:
*/
/************ Begin %include sections from the grammar ************************/
#line 1 "queryparser/queryparser.lemony"

/** @file
 * @brief build a Xapian::Query object from a user query string
 */
/* Copyright (C) 2004-2026 Olly Betts
 * Copyright (C) 2007,2008,2009 Lemur Consulting Ltd
 * Copyright (C) 2010 Adam Sjøgren
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include <config.h>

#include "queryparser_internal.h"

#include "api/queryinternal.h"
#include "omassert.h"
#include "str.h"
#include "stringutils.h"
#include "xapian/error.h"
#include "xapian/unicode.h"

// Include the list of token values lemon generates.
#include "queryparser_token.h"

#include "word-breaker.h"

#include <algorithm>
#include <cstring>
#include <limits>
#include <list>
#include <string>
#include <string_view>
#include <vector>

// We create the yyParser on the stack.
#define Parse_ENGINEALWAYSONSTACK

using namespace std;

using namespace Xapian;

static constexpr unsigned NO_EDIT_DISTANCE = unsigned(-1);
static constexpr unsigned DEFAULT_EDIT_DISTANCE = 2;

inline bool
U_isupper(unsigned ch) {
    return ch < 128 && C_isupper(static_cast<unsigned char>(ch));
}

inline bool
U_isdigit(unsigned ch) {
    return ch < 128 && C_isdigit(static_cast<unsigned char>(ch));
}

inline bool
U_isalpha(unsigned ch) {
    return ch < 128 && C_isalpha(static_cast<unsigned char>(ch));
}

using Xapian::Unicode::is_whitespace;

inline bool
is_not_whitespace(unsigned ch) {
    return !is_whitespace(ch);
}

using Xapian::Unicode::is_wordchar;

inline bool
is_not_wordchar(unsigned ch) {
    return !is_wordchar(ch);
}

inline bool
is_digit(unsigned ch) {
    return (Unicode::get_category(ch) == Unicode::DECIMAL_DIGIT_NUMBER);
}

// FIXME: we used to keep trailing "-" (e.g. Cl-) but it's of dubious utility
// and there's the risk of hyphens getting stuck onto the end of terms...
//
// There are currently assumptions below that this only matches ASCII
// characters.
inline bool
is_suffix(unsigned ch) {
    return ch == '+' || ch == '#';
}

inline bool
is_double_quote(unsigned ch) {
    // We simply treat all double quotes as equivalent, which is a bit crude,
    // but it isn't clear that it would actually better to require them to
    // match up exactly.
    //
    // 0x201c is Unicode opening double quote.
    // 0x201d is Unicode closing double quote.
    return ch == '"' || ch == 0x201c || ch == 0x201d;
}

inline bool
prefix_needs_colon(const string & prefix, unsigned ch)
{
    if (!U_isupper(ch) && ch != ':') return false;
    string::size_type len = prefix.length();
    return (len > 1 && prefix[len - 1] != ':');
}

using Unicode::is_currency;

inline bool
is_positional(Xapian::Query::op op)
{
    return (op == Xapian::Query::OP_PHRASE || op == Xapian::Query::OP_NEAR);
}

class Terms;

/** Class used to pass information about a token from lexer to parser.
 *
 *  Generally an instance of this class carries term information, but it can be
 *  used for a range query, and with some operators (e.g. the distance in
 *  NEAR/3 or ADJ/3, etc).
 */
class Term {
    State * state;

  public:
    string name;
    const FieldInfo * field_info;
    string unstemmed;
    QueryParser::stem_strategy stem;
    termpos pos;
    Query query;
    unsigned edit_distance;

    Term(const string &name_, termpos pos_)
	: name(name_), stem(QueryParser::STEM_NONE), pos(pos_) { }
    explicit Term(const string &name_)
	: name(name_), stem(QueryParser::STEM_NONE), pos(0) { }
    Term(const string &name_, const FieldInfo * field_info_)
	: name(name_), field_info(field_info_),
	  stem(QueryParser::STEM_NONE), pos(0) { }
    explicit Term(termpos pos_) : stem(QueryParser::STEM_NONE), pos(pos_) { }
    Term(State * state_, const string &name_, const FieldInfo * field_info_,
	 const string &unstemmed_,
	 QueryParser::stem_strategy stem_ = QueryParser::STEM_NONE,
	 termpos pos_ = 0,
	 unsigned edit_distance_ = NO_EDIT_DISTANCE)
	: state(state_), name(name_), field_info(field_info_),
	  unstemmed(unstemmed_), stem(stem_), pos(pos_),
	  edit_distance(edit_distance_) { }
    // For RANGE tokens.
    Term(const Xapian::Query & q, const string & grouping)
	: name(grouping), query(q) { }

    string make_term(const string & prefix) const;

    void need_positions() {
	if (stem == QueryParser::STEM_SOME) stem = QueryParser::STEM_NONE;
    }

    termpos get_termpos() const { return pos; }

    string get_grouping() const {
	return field_info->grouping;
    }

    Query * as_fuzzy_query(State * state) const;

    Query * as_wildcarded_query(State * state) const;

    /** Build a query for a term at the very end of the query string when
     *  FLAG_PARTIAL is in use.
     *
     *  This query should match documents containing any terms which start with
     *  the characters specified, but should give a higher score to exact
     *  matches (since the user might have finished typing - we simply don't
     *  know).
     */
    Query * as_partial_query(State * state_) const;

    /** Build a query for a string of words without explicit word breaks. */
    Query* as_unbroken_query() const;

    /** Handle text without explicit word breaks in a positional context. */
    void as_positional_unbroken(Terms* terms) const;

    /// Range query.
    Query as_range_query() const;

    Query get_query() const;

    Query get_query_with_synonyms() const;

    Query get_query_with_auto_synonyms() const;
};

/// Parser State shared between the lexer and the parser.
class State {
    QueryParser::Internal * qpi;

  public:
    Query query;
    const char* error = NULL;
    unsigned flags;
    unsigned int should_stem_mask =
	(1 << Unicode::LOWERCASE_LETTER) |
	(1 << Unicode::TITLECASE_LETTER) |
	(1 << Unicode::MODIFIER_LETTER) |
	(1 << Unicode::OTHER_LETTER);
    Query::op effective_default_op;

    State(QueryParser::Internal * qpi_, unsigned flags_)
	: qpi(qpi_), flags(flags_), effective_default_op(qpi_->default_op)
    {
	if ((flags & QueryParser::FLAG_NO_PROPER_NOUN_HEURISTIC) ||
	    qpi->stemmer.is_none() ||
	    !qpi->stemmer.internal->use_proper_noun_heuristic()) {
	    should_stem_mask |= (1 << Unicode::UPPERCASE_LETTER);
	}
	if ((flags & QueryParser::FLAG_NO_POSITIONS)) {
	    if (is_positional(effective_default_op)) {
		effective_default_op = Query::OP_AND;
	    }
	}
    }

    string stem_term(const string &term) {
	return qpi->stemmer(term);
    }

    void add_to_stoplist(const Term * term) {
	qpi->stoplist.push_back(term->name);
    }

    void add_to_unstem(const string & term, const string & unstemmed) {
	qpi->unstem.insert(make_pair(term, unstemmed));
    }

    Term * range(const string &a, const string &b) {
	for (auto i : qpi->rangeprocs) {
	    Xapian::Query range_query = (i.proc)->check_range(a, b);
	    Xapian::Query::op op = range_query.get_type();
	    switch (op) {
		case Xapian::Query::OP_INVALID:
		    break;
		case Xapian::Query::OP_VALUE_RANGE:
		case Xapian::Query::OP_VALUE_GE:
		case Xapian::Query::OP_VALUE_LE:
		    if (i.default_grouping) {
			Xapian::Internal::QueryValueBase * base =
			    static_cast<Xapian::Internal::QueryValueBase*>(
				range_query.internal.get());
			Xapian::valueno slot = base->get_slot();
			return new Term(range_query, str(slot));
		    }
		    // FALLTHRU
		case Xapian::Query::LEAF_TERM:
		    return new Term(range_query, i.grouping);
		default:
		    return new Term(range_query, string());
	    }
	}
	return NULL;
    }

    Query::op default_op() const {
	return effective_default_op;
    }

    bool is_stopword(const Term *term) const {
	return qpi->stopper && (*qpi->stopper)(term->name);
    }

    Database get_database() const {
	return qpi->db;
    }

    const Stopper * get_stopper() const {
	return qpi->stopper.get();
    }

    QueryParser::stop_strategy get_stopper_strategy() const {
	return qpi->stop_mode;
    }

    size_t stoplist_size() const {
	return qpi->stoplist.size();
    }

    void stoplist_resize(size_t s) {
	qpi->stoplist.resize(s);
    }

    Xapian::termcount get_max_wildcard_expansion() const {
	return qpi->max_wildcard_expansion;
    }

    int get_max_wildcard_type() const {
	return qpi->max_wildcard_type;
    }

    unsigned get_min_wildcard_prefix_len() const {
	return qpi->min_wildcard_prefix_len;
    }

    Xapian::termcount get_max_partial_expansion() const {
	return qpi->max_partial_expansion;
    }

    int get_max_partial_type() const {
	return qpi->max_partial_type;
    }

    unsigned get_min_partial_prefix_len() const {
	return qpi->min_partial_prefix_len;
    }

    Xapian::termcount get_max_fuzzy_expansion() const {
	return qpi->max_fuzzy_expansion;
    }

    int get_max_fuzzy_type() const {
	return qpi->max_fuzzy_type;
    }
};

string
Term::make_term(const string & prefix) const
{
    if (state->get_stopper_strategy() == QueryParser::STOP_ALL) {
	const Xapian::Stopper* stopper = state->get_stopper();
	if (stopper && (*stopper)(name)) {
	    return string();
	}
    }

    string term;
    if (stem != QueryParser::STEM_NONE && stem != QueryParser::STEM_ALL)
	term += 'Z';
    if (!prefix.empty()) {
	term += prefix;
	if (prefix_needs_colon(prefix, name[0])) term += ':';
    }
    if (stem != QueryParser::STEM_NONE) {
	term += state->stem_term(name);
    } else {
	term += name;
    }

    if (!unstemmed.empty())
	state->add_to_unstem(term, unstemmed);
    return term;
}

// Iterator shim to allow building a synonym query from a TermIterator pair.
class SynonymIterator {
    Xapian::TermIterator i;

    Xapian::termpos pos;

    const Xapian::Query * first;

  public:
    SynonymIterator(const Xapian::TermIterator & i_,
		    Xapian::termpos pos_ = 0,
		    const Xapian::Query * first_ = NULL)
	: i(i_), pos(pos_), first(first_) { }

    SynonymIterator & operator++() {
	if (first)
	    first = NULL;
	else
	    ++i;
	return *this;
    }

    const Xapian::Query operator*() const {
	if (first) return *first;
	return Xapian::Query(*i, 1, pos);
    }

    bool operator==(const SynonymIterator & o) const {
	return i == o.i && first == o.first;
    }

    bool operator!=(const SynonymIterator & o) const {
	return !(*this == o);
    }

    typedef std::input_iterator_tag iterator_category;
    typedef Xapian::Query value_type;
    typedef Xapian::termcount_diff difference_type;
    typedef Xapian::Query * pointer;
    typedef Xapian::Query & reference;
};

Query
Term::get_query_with_synonyms() const
{
    // Handle single-word synonyms with each prefix.
    const auto& prefixes = field_info->prefixes;
    if (prefixes.empty()) {
	Assert(field_info->proc);
	return (*field_info->proc)(name);
    }

    Query q = get_query();

    for (auto&& prefix : prefixes) {
	// First try the unstemmed term:
	string term;
	if (!prefix.empty()) {
	    term += prefix;
	    if (prefix_needs_colon(prefix, name[0])) term += ':';
	}
	term += name;

	Xapian::Database db = state->get_database();
	Xapian::TermIterator syn = db.synonyms_begin(term);
	Xapian::TermIterator end = db.synonyms_end(term);
	if (syn == end && stem != QueryParser::STEM_NONE) {
	    // If that has no synonyms, try the stemmed form:
	    term = 'Z';
	    if (!prefix.empty()) {
		term += prefix;
		if (prefix_needs_colon(prefix, name[0])) term += ':';
	    }
	    term += state->stem_term(name);
	    syn = db.synonyms_begin(term);
	    end = db.synonyms_end(term);
	}
	q = Query(q.OP_SYNONYM,
		  SynonymIterator(syn, pos, &q),
		  SynonymIterator(end));
    }
    return q;
}

Query
Term::get_query_with_auto_synonyms() const
{
    const unsigned MASK_ENABLE_AUTO_SYNONYMS =
	QueryParser::FLAG_AUTO_SYNONYMS |
	QueryParser::FLAG_AUTO_MULTIWORD_SYNONYMS;
    if (state->flags & MASK_ENABLE_AUTO_SYNONYMS)
	return get_query_with_synonyms();

    return get_query();
}

static void
add_to_query(Query *& q, Query::op op, Query * term)
{
    Assert(term);
    if (q) {
	if (op == Query::OP_OR) {
	    *q |= *term;
	} else if (op == Query::OP_AND) {
	    *q &= *term;
	} else {
	    *q = Query(op, *q, *term);
	}
	delete term;
    } else {
	q = term;
    }
}

static void
add_to_query(Query *& q, Query::op op, const Query & term)
{
    if (q) {
	if (op == Query::OP_OR) {
	    *q |= term;
	} else if (op == Query::OP_AND) {
	    *q &= term;
	} else {
	    *q = Query(op, *q, term);
	}
    } else {
	q = new Query(term);
    }
}

Query
Term::get_query() const
{
    const auto& prefixes = field_info->prefixes;
    if (prefixes.empty()) {
	Assert(field_info->proc);
	return (*field_info->proc)(name);
    }
    auto piter = prefixes.begin();
    const string& term = make_term(*piter);
    if (term.empty()) return Query();
    Query q(term, 1, pos);
    while (++piter != prefixes.end()) {
	q |= Query(make_term(*piter), 1, pos);
    }
    return q;
}

Query *
Term::as_fuzzy_query(State* state_) const
{
    const auto& prefixes = field_info->prefixes;
    Xapian::termcount max = state_->get_max_fuzzy_expansion();
    int query_flags = state_->get_max_fuzzy_type();
    vector<Query> subqs;
    subqs.reserve(prefixes.size());
    for (auto&& prefix : prefixes) {
	// Combine with OP_OR, and apply OP_SYNONYM afterwards.
	subqs.emplace_back(Query::OP_EDIT_DISTANCE,
			   prefix + name,
			   max,
			   query_flags,
			   Query::OP_OR,
			   edit_distance,
			   prefix.size());
    }
    Query* q = new Query(Query::OP_SYNONYM, subqs.begin(), subqs.end());
    delete this;
    return q;
}

Query *
Term::as_wildcarded_query(State * state_) const
{
    const auto& prefixes = field_info->prefixes;
    Xapian::termcount max = state_->get_max_wildcard_expansion();
    int query_flags = state_->get_max_wildcard_type();
    if (state_->flags & QueryParser::FLAG_WILDCARD_SINGLE)
	query_flags |= Query::WILDCARD_PATTERN_SINGLE;
    if (state_->flags & QueryParser::FLAG_WILDCARD_MULTI)
	query_flags |= Query::WILDCARD_PATTERN_MULTI;
    vector<Query> subqs;
    subqs.reserve(prefixes.size());
    for (string root : prefixes) {
	root += name;
	// Combine with OP_OR, and apply OP_SYNONYM afterwards.
	subqs.push_back(Query(Query::OP_WILDCARD, root, max, query_flags,
			      Query::OP_OR));
    }
    Query * q = new Query(Query::OP_SYNONYM, subqs.begin(), subqs.end());
    delete this;
    return q;
}

Query *
Term::as_partial_query(State * state_) const
{
    Xapian::termcount max = state_->get_max_partial_expansion();
    int max_type = state_->get_max_partial_type();
    vector<Query> subqs_partial; // A synonym of all the partial terms.
    vector<Query> subqs_full; // A synonym of all the full terms.

    for (const string& prefix : field_info->prefixes) {
	string root = prefix;
	root += name;
	// Combine with OP_OR, and apply OP_SYNONYM afterwards.
	subqs_partial.push_back(Query(Query::OP_WILDCARD, root, max, max_type,
				      Query::OP_OR));
	if (!state->is_stopword(this)) {
	    // Add the term, as it would normally be handled, as an alternative
	    // (unless it is a stopword).
	    subqs_full.push_back(Query(make_term(prefix), 1, pos));
	}
    }
    Query * q = new Query(Query::OP_OR,
			  Query(Query::OP_SYNONYM,
				subqs_partial.begin(), subqs_partial.end()),
			  Query(Query::OP_SYNONYM,
				subqs_full.begin(), subqs_full.end()));
    delete this;
    return q;
}

Query *
Term::as_unbroken_query() const
{
    const auto& prefixes = field_info->prefixes;
    Query *q;
    vector<Query> prefix_subqs;

#ifdef USE_ICU
    if (state->flags & QueryParser::FLAG_WORD_BREAKS) {
	for (WordIterator tk(name); tk != WordIterator(); ++tk) {
	    const string& token = *tk;
	    for (const string& prefix : prefixes) {
		prefix_subqs.push_back(Query(prefix + token, 1, pos));
	    }
	}

	q = new Query(Query::OP_AND, prefix_subqs.begin(), prefix_subqs.end());

	delete this;
	return q;
    }
#endif

    vector<Query> ngram_subqs;

    for (const string& prefix : prefixes) {
	for (NgramIterator tk(name); tk != NgramIterator(); ++tk) {
	    ngram_subqs.push_back(Query(prefix + *tk, 1, pos));
	}
	prefix_subqs.push_back(Query(Query::OP_AND,
				     ngram_subqs.begin(), ngram_subqs.end()));
	ngram_subqs.clear();
    }
    q = new Query(Query::OP_OR, prefix_subqs.begin(), prefix_subqs.end());

    delete this;
    return q;
}

Query
Term::as_range_query() const
{
    Query q = query;
    delete this;
    return q;
}

inline bool
is_phrase_generator(unsigned ch)
{
    // These characters generate a phrase search.
    // Ordered mostly by frequency of calls to this function done when
    // running the testcases in api_queryparser.cc.
    return (ch && ch < 128 && strchr(".-/:\\@", ch) != NULL);
}

inline bool
is_stem_preventer(unsigned ch)
{
    return (ch && ch < 128 && strchr("(/\\@<>=*[{\"", ch) != NULL);
}

inline bool
should_stem(const string& term, const State& state)
{
    Utf8Iterator u(term);
    return ((state.should_stem_mask >> Unicode::get_category(*u)) & 1);
}

/** Value representing "ignore this" when returned by check_infix() or
 *  check_infix_digit().
 */
const unsigned UNICODE_IGNORE = numeric_limits<unsigned>::max();

inline unsigned check_infix(unsigned ch) {
    if (ch == '\'' || ch == '&' || ch == 0xb7 || ch == 0x5f4 || ch == 0x2027) {
	// Unicode includes all these except '&' in its word boundary rules,
	// as well as 0x2019 (which we handle below) and ':' (for Swedish
	// apparently, but we ignore this for now as it's problematic in
	// real world cases:
	// https://en.wikipedia.org/wiki/Colon_(punctuation)#Usage_in_other_languages
	// ).
	return ch;
    }
    if (ch >= 0x200c) {
	// 0x2019 is Unicode apostrophe and single closing quote.
	// 0x201b is Unicode single opening quote with the tail rising.
	if (ch == 0x2019 || ch == 0x201b)
	    return '\'';
	// 0x200c and 0x200d are zero width non-joiner and joiner respectively.
	// 0x2060 and 0xfeff are word joiners (0xfeff deprecated since Unicode
	// 3.2).
	if (ch <= 0x200d || ch == 0x2060 || ch == 0xfeff)
	    return UNICODE_IGNORE;
    }
    // 0xad is SOFT HYPHEN which marks a potential hyphenation point in a word.
    if (ch == 0xad)
	return UNICODE_IGNORE;
    return 0;
}

inline unsigned check_infix_digit(unsigned ch) {
    // This list of characters comes from Unicode's word identifying algorithm.
    switch (ch) {
	case ',':
	case '.':
	case ';':
	case 0x037e: // GREEK QUESTION MARK
	case 0x0589: // ARMENIAN FULL STOP
	case 0x060D: // ARABIC DATE SEPARATOR
	case 0x07F8: // NKO COMMA
	case 0x2044: // FRACTION SLASH
	case 0xFE10: // PRESENTATION FORM FOR VERTICAL COMMA
	case 0xFE13: // PRESENTATION FORM FOR VERTICAL COLON
	case 0xFE14: // PRESENTATION FORM FOR VERTICAL SEMICOLON
	    return ch;
    }
    if (ch >= 0x200b && (ch <= 0x200d || ch == 0x2060 || ch == 0xfeff))
	return UNICODE_IGNORE;
    return 0;
}

// Prototype a function lemon generates, but which we want to call before that
// in the generated source code file.
struct yyParser;
static void yy_parse_failed(yyParser *);

void
QueryParser::Internal::add_prefix(string_view field, string_view prefix)
{
    // Allow for optional trailing `:` for consistency with how range prefixes
    // are specified.
    if (!field.empty() && field.back() == ':') {
	field = field.substr(0, field.size() - 1);
    }
#ifdef  __cpp_lib_associative_heterogeneous_insertion // C++26
    auto [it, inserted] = field_map.try_emplace(field, NON_BOOLEAN);
#else
    auto [it, inserted] = field_map.try_emplace(string(field), NON_BOOLEAN);
#endif
    auto&& p = it->second;
    if (inserted) {
	p.append(prefix);
	return;
    }

    // Check that this is the same type of filter as the existing one(s).
    if (p.type != NON_BOOLEAN) {
	throw Xapian::InvalidOperationError("Can't use add_prefix() and "
					    "add_boolean_prefix() on the "
					    "same field name, or "
					    "add_boolean_prefix() with "
					    "different values of the "
					    "'exclusive' parameter");
    }
    if (p.proc)
	throw Xapian::FeatureUnavailableError("Mixing FieldProcessor objects "
					      "and string prefixes currently "
					      "not supported");
    // Only add if it's not already there as duplicate entries just result
    // in redundant query terms.  This is a linear scan so makes calling
    // add_prefix() n times for the same field value with different prefix
    // values O(n²), but you wouldn't realistically want to map one field
    // to more than a handful of prefixes.
    auto& prefixes = p.prefixes;
    if (find(prefixes.begin(), prefixes.end(), prefix) == prefixes.end()) {
	p.append(prefix);
    }
}

void
QueryParser::Internal::add_prefix(string_view field, FieldProcessor* proc)
{
    // Allow for optional trailing `:` for consistency with how range prefixes
    // are specified.
    if (!field.empty() && field.back() == ':') {
	field = field.substr(0, field.size() - 1);
    }
#ifdef  __cpp_lib_associative_heterogeneous_insertion // C++26
    auto [it, inserted] = field_map.try_emplace(field,
						NON_BOOLEAN, proc);
#else
    auto [it, inserted] = field_map.try_emplace(string(field),
						NON_BOOLEAN, proc);
#endif
    if (inserted)
	return;

    auto&& p = it->second;
    // Check that this is the same type of filter as the existing one(s).
    if (p.type != NON_BOOLEAN) {
	throw Xapian::InvalidOperationError("Can't use add_prefix() and "
					    "add_boolean_prefix() on the "
					    "same field name, or "
					    "add_boolean_prefix() with "
					    "different values of the "
					    "'exclusive' parameter");
    }
    if (!p.prefixes.empty())
	throw Xapian::FeatureUnavailableError("Mixing FieldProcessor objects "
					      "and string prefixes currently "
					      "not supported");
    throw Xapian::FeatureUnavailableError("Multiple FieldProcessor objects "
					  "for the same prefix currently not "
					  "supported");
}

void
QueryParser::Internal::add_boolean_prefix(string_view field,
					  string_view prefix,
					  const string* grouping)
{
    // Allow for optional trailing `:` for consistency with how range prefixes
    // are specified.
    if (!field.empty() && field.back() == ':') {
	field = field.substr(0, field.size() - 1);
    }
    // Don't allow the empty prefix to be set as boolean as it doesn't
    // really make sense.
    if (field.empty())
	throw Xapian::UnimplementedError("Can't set the empty prefix to be a boolean filter");
    // If grouping == nullptr then it defaults to field which is not empty().
    bool inclusive = grouping && grouping->empty();
    filter_type type = inclusive ? BOOLEAN : BOOLEAN_EXCLUSIVE;
#ifdef __cpp_lib_associative_heterogeneous_insertion // C++26
    auto [it, inserted] = field_map.try_emplace(field, type,
						grouping ? *grouping : field);
#else
    auto [it, inserted] = field_map.try_emplace(string(field), type,
						grouping ? *grouping : field);
#endif
    auto&& p = it->second;
    if (inserted) {
	p.append(prefix);
	return;
    }

    // Check that this is the same type of filter as the existing one(s).
    if (p.type != type) {
	throw Xapian::InvalidOperationError("Can't use add_prefix() and "
					    "add_boolean_prefix() on the "
					    "same field name, or "
					    "add_boolean_prefix() with "
					    "different values of the "
					    "'exclusive' parameter"); // FIXME
    }
    if (p.proc)
	throw Xapian::FeatureUnavailableError("Mixing FieldProcessor objects "
					      "and string prefixes currently "
					      "not supported");
    // Only add if it's not already there as duplicate entries just result
    // in redundant query terms.  This is a linear scan so makes calling
    // add_prefix() n times for the same field value with different prefix
    // values O(n²), but you wouldn't realistically want to map one field
    // to more than a handful of prefixes.
    auto& prefixes = p.prefixes;
    if (find(prefixes.begin(), prefixes.end(), prefix) == prefixes.end()) {
	prefixes.emplace_back(prefix); // FIXME grouping
    }
}

void
QueryParser::Internal::add_boolean_prefix(string_view field,
					  FieldProcessor *proc,
					  const string* grouping)
{
    // Allow for optional trailing `:` for consistency with how range prefixes
    // are specified.
    if (!field.empty() && field.back() == ':') {
	field = field.substr(0, field.size() - 1);
    }
    // Don't allow the empty prefix to be set as boolean as it doesn't
    // really make sense.
    if (field.empty())
	throw Xapian::UnimplementedError("Can't set the empty prefix to be a boolean filter");
    // If grouping == nullptr then it defaults to field which is not empty().
    bool inclusive = grouping && grouping->empty();
    filter_type type = inclusive ? BOOLEAN : BOOLEAN_EXCLUSIVE;
#ifdef __cpp_lib_associative_heterogeneous_insertion // C++26
    auto [it, inserted] = field_map.try_emplace(field, type, proc,
						grouping ? *grouping : field);
#else
    auto [it, inserted] = field_map.try_emplace(string(field), type, proc,
						grouping ? *grouping : field);
#endif
    if (inserted)
	return;

    auto&& p = it->second;
    // Check that this is the same type of filter as the existing one(s).
    if (p.type != type) {
	throw Xapian::InvalidOperationError("Can't use add_prefix() and "
					    "add_boolean_prefix() on the "
					    "same field name, or "
					    "add_boolean_prefix() with "
					    "different values of the "
					    "'exclusive' parameter"); // FIXME
    }
    if (!p.prefixes.empty())
	throw Xapian::FeatureUnavailableError("Mixing FieldProcessor objects "
					      "and string prefixes currently "
					      "not supported");
    throw Xapian::FeatureUnavailableError("Multiple FieldProcessor objects "
					  "for the same prefix currently not "
					  "supported");
}

inline bool
is_extended_wildcard(unsigned ch, unsigned flags)
{
    if (ch == '*') return (flags & QueryParser::FLAG_WILDCARD_MULTI);
    if (ch == '?') return (flags & QueryParser::FLAG_WILDCARD_SINGLE);
    return false;
}

string
QueryParser::Internal::parse_term(Utf8Iterator& it, const Utf8Iterator& end,
				  bool try_word_break, unsigned flags,
				  bool& needs_word_break, bool& was_acronym,
				  size_t& first_wildcard,
				  size_t& char_count,
				  unsigned& edit_distance)
{
    string term;
    char_count = 0;
    // Look for initials separated by '.' (e.g. P.T.O., U.N.C.L.E).
    // Don't worry if there's a trailing '.' or not.
    if (U_isupper(*it)) {
	string t;
	Utf8Iterator p = it;
	do {
	    Unicode::append_utf8(t, *p++);
	    ++char_count;
	} while (p != end && *p == '.' && ++p != end && U_isupper(*p));
	// One letter does not make an acronym!  If we handled a single
	// uppercase letter here, we wouldn't catch M&S below.
	if (t.length() > 1) {
	    // Check there's not a (lower case) letter or digit
	    // immediately after it.
	    // FIXME: should I.B.M..P.T.O be a range search?
	    if (p == end || !is_wordchar(*p)) {
		it = p;
		swap(term, t);
	    } else {
		char_count = 0;
	    }
	}
    }
    was_acronym = !term.empty();

    if (try_word_break && term.empty() && is_unbroken_script(*it)) {
	const char* start = it.raw();
	char_count = get_unbroken(it);
	term.assign(start, it.raw() - start);
	needs_word_break = true;
    }

    if (term.empty()) {
	unsigned prevch = *it;
	if (first_wildcard == term.npos &&
	    is_extended_wildcard(prevch, flags)) {
	    // Leading wildcard.
	    first_wildcard = 0;
	}
	Unicode::append_utf8(term, prevch);
	char_count = 1;
	while (++it != end) {
	    if (try_word_break && is_unbroken_script(*it)) break;
	    unsigned ch = *it;
	    if (is_extended_wildcard(ch, flags)) {
		if (first_wildcard == term.npos) {
		    first_wildcard = char_count;
		}
	    } else if (!is_wordchar(ch)) {
		// Treat a single embedded '&' or "'" or similar as a word
		// character (e.g. AT&T, Fred's).  Also, normalise
		// apostrophes to ASCII apostrophe.
		Utf8Iterator p = it;
		++p;
		if (p == end) break;
		unsigned nextch = *p;
		if (is_extended_wildcard(nextch, flags)) {
		    // A wildcard follows, which could expand to a digit or a non-digit.
		    unsigned ch_orig = ch;
		    ch = check_infix(ch);
		    if (!ch && is_digit(prevch))
			ch = check_infix_digit(ch_orig);
		    if (!ch)
			break;
		} else {
		    if (!is_wordchar(nextch)) break;
		}
		if (is_digit(prevch) && is_digit(nextch)) {
		    ch = check_infix_digit(ch);
		} else {
		    ch = check_infix(ch);
		}
		if (!ch) break;
		if (ch == UNICODE_IGNORE)
		    continue;
	    }
	    Unicode::append_utf8(term, ch);
	    ++char_count;
	    prevch = ch;
	}
	if (it != end && is_suffix(*it)) {
	    string suff_term = term;
	    Utf8Iterator p = it;
	    // Keep trailing + (e.g. C++, Na+) or # (e.g. C#).
	    do {
		// Assumes is_suffix() only matches ASCII.
		if (suff_term.size() - term.size() == 3) {
		    suff_term.resize(0);
		    break;
		}
		suff_term += *p;
	    } while (is_suffix(*++p));
	    if (!suff_term.empty() && (p == end || !is_wordchar(*p))) {
		// If the suffixed term doesn't exist, check that the
		// non-suffixed term does.  This also takes care of
		// the case when QueryParser::set_database() hasn't
		// been called.
		bool use_suff_term = false;
		string lc = Unicode::tolower(suff_term);
		if (db.term_exists(lc)) {
		    use_suff_term = true;
		} else {
		    lc = Unicode::tolower(term);
		    if (!db.term_exists(lc)) use_suff_term = true;
		}
		if (use_suff_term) {
		    // Assumes is_suffix() only matches ASCII.
		    char_count += (suff_term.size() - term.size());
		    term = suff_term;
		    it = p;
		}
	    }
	}
	if (first_wildcard == term.npos &&
	    (flags & QueryParser::FLAG_WILDCARD)) {
	    // Check for right-truncation.
	    if (it != end && *it == '*') {
		++it;
		first_wildcard = char_count;
	    }
	}
	if (it != end &&
	    (flags & QueryParser::FLAG_FUZZY) &&
	    // Not a wildcard.
	    first_wildcard == string::npos &&
	    *it == '~') {
	    Utf8Iterator p = it;
	    ++p;
	    unsigned ch = *p;
	    if (p == end || is_whitespace(ch) || ch == ')') {
		it = p;
		edit_distance = DEFAULT_EDIT_DISTANCE;
	    } else if (U_isdigit(ch)) {
		unsigned distance = ch - '0';
		while (++p != end && U_isdigit(*p)) {
		    distance = distance * 10 + (*p - '0');
		}
		if (p != end && *p == '.') {
		    if (distance == 0) goto fractional;
		    // Ignore the fractional part on e.g. foo~12.5
		    while (++p != end && U_isdigit(*p)) { }
		}
		if (p == end || is_whitespace(ch) || ch == ')') {
		    it = p;
		    edit_distance = distance;
		}
	    } else if (ch == '.') {
fractional:
		double fraction = 0.0;
		double digit = 0.1;
		while (++p != end && U_isdigit(*p)) {
		    fraction += digit * (*p - '0');
		    digit *= 0.1;
		}
		if (p == end || is_whitespace(ch) || ch == ')') {
		    it = p;
		    unsigned codepoints = 0;
		    for (Utf8Iterator u8(term); u8 != Utf8Iterator(); ++u8) {
			++codepoints;
		    }
		    edit_distance = unsigned(codepoints * fraction);
		}
	    }
	}
    }
    return term;
}

#line 1772 "queryparser/queryparser.lemony"


struct ProbQuery {
    Query* query = NULL;
    Query* love = NULL;
    Query* hate = NULL;
    // filter is a map from prefix to a query for that prefix.  Queries with
    // the same prefix are combined with OR, and the results of this are
    // combined with AND to get the full filter.
    map<string, Query> filter;

    ProbQuery() {}

    explicit
    ProbQuery(Query* query_) : query(query_) {}

    ~ProbQuery() {
	delete query;
	delete love;
	delete hate;
    }

    void add_filter(const string& grouping, const Query & q) {
	filter[grouping] = q;
    }

    void append_filter(const string& grouping, const Query & qnew) {
	auto it = filter.find(grouping);
	if (it == filter.end()) {
	    filter.insert(make_pair(grouping, qnew));
	} else {
	    Query & q = it->second;
	    // We OR multiple filters with the same prefix if they're
	    // exclusive, otherwise we AND them.
	    bool exclusive = !grouping.empty();
	    if (exclusive) {
		q |= qnew;
	    } else {
		q &= qnew;
	    }
	}
    }

    void add_filter_range(const string& grouping, const Query & range) {
	filter[grouping] = range;
    }

    void append_filter_range(const string& grouping, const Query & range) {
	Query & q = filter[grouping];
	q |= range;
    }

    Query merge_filters() const {
	auto i = filter.begin();
	Assert(i != filter.end());
	Query q = i->second;
	while (++i != filter.end()) {
	    q &= i->second;
	}
	return q;
    }
};

/// A group of terms separated only by whitespace.
class TermGroup {
    vector<Term *> terms;

    /** Controls how to handle a group where all terms are stopwords.
     *
     *  If true, then as_group() returns NULL.  If false, then the
     *  stopword status of the terms is ignored.
     */
    bool empty_ok;

    TermGroup(Term* t1, Term* t2) : empty_ok(false) {
	add_term(t1);
	add_term(t2);
    }

  public:
    /// Factory function - ensures heap allocation.
    static TermGroup* create(Term* t1, Term* t2) {
	return new TermGroup(t1, t2);
    }

    ~TermGroup() {
	for (auto&& t : terms) {
	    delete t;
	}
    }

    /// Add a Term object to this TermGroup object.
    void add_term(Term * term) {
	terms.push_back(term);
    }

    /// Set the empty_ok flag.
    void set_empty_ok() { empty_ok = true; }

    /// Convert to a Xapian::Query * using default_op.
    Query * as_group(State *state) const;
};

Query *
TermGroup::as_group(State *state) const
{
    const Xapian::Stopper * stopper = state->get_stopper();
    size_t stoplist_size = state->stoplist_size();
    bool default_op_is_positional = is_positional(state->default_op());
reprocess:
    Query::op default_op = state->default_op();
    vector<Query> subqs;
    subqs.reserve(terms.size());
    if (state->flags & QueryParser::FLAG_AUTO_MULTIWORD_SYNONYMS) {
	// Check for multi-word synonyms.
	Database db = state->get_database();

	string key;
	vector<Term*>::size_type begin = 0;
	vector<Term*>::size_type i = begin;
	while (terms.size() - i > 0) {
	    size_t longest_match = 0;
	    // This value is never used, but GCC 4.8 warns with
	    // -Wmaybe-uninitialized (GCC 5.4 doesn't).
	    vector<Term*>::size_type longest_match_end = 0;
	    if (terms.size() - i >= 2) {
		// Greedily try to match as many consecutive words as possible.
		key = terms[i]->name;
		key += ' ';
		key += terms[i + 1]->name;
		TermIterator synkey(db.synonym_keys_begin(key));
		TermIterator synend(db.synonym_keys_end(key));
		if (synkey != synend) {
		    longest_match = key.size();
		    longest_match_end = i + 2;
		    for (auto j = i + 2; j < terms.size(); ++j) {
			key += ' ';
			key += terms[j]->name;
			synkey.skip_to(key);
			if (synkey == synend)
			    break;
			const string& found = *synkey;
			if (!startswith(found, key))
			    break;
			if (found.size() == key.size()) {
			    longest_match = key.size();
			    longest_match_end = j + 1;
			}
		    }
		}
	    }
	    if (longest_match == 0) {
		// No multi-synonym matches at position i.
		if (stopper && (*stopper)(terms[i]->name)) {
		    state->add_to_stoplist(terms[i]);
		} else {
		    if (default_op_is_positional)
			terms[i]->need_positions();
		    subqs.push_back(terms[i]->get_query_with_auto_synonyms());
		}
		begin = ++i;
		continue;
	    }
	    i = longest_match_end;
	    key.resize(longest_match);

	    vector<Query> subqs2;
	    for (auto j = begin; j != i; ++j) {
		if (stopper && (*stopper)(terms[j]->name)) {
		    state->add_to_stoplist(terms[j]);
		} else {
		    if (default_op_is_positional)
			terms[i]->need_positions();
		    subqs2.push_back(terms[j]->get_query());
		}
	    }
	    Query q_original_terms;
	    if (default_op_is_positional) {
		q_original_terms = Query(default_op,
					 subqs2.begin(), subqs2.end(),
					 subqs2.size() + 9);
	    } else {
		q_original_terms = Query(default_op,
					 subqs2.begin(), subqs2.end());
	    }
	    subqs2.clear();

	    // Use the position of the first term for the synonyms.
	    TermIterator syn = db.synonyms_begin(key);
	    Query q(Query::OP_SYNONYM,
		    SynonymIterator(syn, terms[begin]->pos, &q_original_terms),
		    SynonymIterator(db.synonyms_end(key)));
	    subqs.push_back(q);

	    begin = i;
	}
    } else {
	vector<Term*>::const_iterator i;
	for (i = terms.begin(); i != terms.end(); ++i) {
	    if (stopper && (*stopper)((*i)->name)) {
		state->add_to_stoplist(*i);
	    } else {
		if (default_op_is_positional)
		    (*i)->need_positions();
		subqs.push_back((*i)->get_query_with_auto_synonyms());
	    }
	}
    }

    if (!empty_ok && stopper &&
	state->get_stopper_strategy() != QueryParser::STOP_ALL &&
	subqs.empty() &&
	stoplist_size < state->stoplist_size()) {
	// This group is all stopwords, so roll-back, disable stopper
	// temporarily, and reprocess this group.
	state->stoplist_resize(stoplist_size);
	stopper = NULL;
	goto reprocess;
    }

    Query * q = NULL;
    if (!subqs.empty()) {
	if (default_op_is_positional) {
	    q = new Query(default_op, subqs.begin(), subqs.end(),
			     subqs.size() + 9);
	} else {
	    q = new Query(default_op, subqs.begin(), subqs.end());
	}
    }
    delete this;
    return q;
}

/// Some terms which form a positional sub-query.
class Terms {
    vector<Term *> terms;

    /** Window size.
     *
     *  size_t(-1) means don't use positional info (so an OP_AND query gets
     *  created).
     */
    size_t window;

    /** Keep track of whether the terms added all have the same list of
     *  prefixes.  If so, we'll build a set of phrases, one using each prefix.
     *  This works around the limitation that a phrase cannot have multiple
     *  components which are "OR" combinations of terms, but is also probably
     *  what users expect: i.e., if a user specifies a phrase in a field, and
     *  that field maps to multiple prefixes, the user probably wants a phrase
     *  returned with all terms having one of those prefixes, rather than a
     *  phrase comprised of terms with differing prefixes.
     */
    bool uniform_prefixes;

    /** The list of prefixes of the terms added.
     *  This will be NULL if the terms have different prefixes.
     */
    const vector<string>* prefixes;

    Query opwindow_subq(Query::op op,
			const vector<Query>& v,
			Xapian::termcount w) const {
	if (op == Query::OP_AND) {
	    return Query(op, v.begin(), v.end());
	}
	return Query(op, v.begin(), v.end(), w);
    }

    /// Convert to a query using the given operator and window size.
    Query * as_opwindow_query(Query::op op, Xapian::termcount w_delta) const {
	if (window == size_t(-1)) op = Query::OP_AND;
	Query * q = NULL;
	size_t n_terms = terms.size();
	Xapian::termcount w = w_delta + terms.size();
	if (uniform_prefixes) {
	    if (prefixes) {
		for (auto&& prefix : *prefixes) {
		    vector<Query> subqs;
		    subqs.reserve(n_terms);
		    for (Term* t : terms) {
			const string& term = t->make_term(prefix);
			if (term.empty()) continue;
			subqs.push_back(Query(term, 1, t->pos));
		    }
		    add_to_query(q, Query::OP_OR, opwindow_subq(op, subqs, w));
		}
	    }
	} else {
	    vector<Query> subqs;
	    subqs.reserve(n_terms);
	    for (Term* t : terms) {
		Query query = t->get_query();
		if (query.get_type() == query.LEAF_MATCH_ALL) continue;
		subqs.push_back(query);
	    }
	    q = new Query(opwindow_subq(op, subqs, w));
	}

	delete this;
	return q;
    }

    explicit Terms(bool no_pos)
	: window(no_pos ? size_t(-1) : 0),
	  uniform_prefixes(true),
	  prefixes(NULL) { }

  public:
    /// Factory function - ensures heap allocation.
    static Terms* create(State* state) {
	return new Terms(state->flags & QueryParser::FLAG_NO_POSITIONS);
    }

    ~Terms() {
	for (auto&& t : terms) {
	    delete t;
	}
    }

    /// Add an unstemmed Term object to this Terms object.
    void add_positional_term(Term * term) {
	const auto& term_prefixes = term->field_info->prefixes;
	if (terms.empty()) {
	    prefixes = &term_prefixes;
	} else if (uniform_prefixes && prefixes != &term_prefixes) {
	    if (*prefixes != term_prefixes)  {
		prefixes = NULL;
		uniform_prefixes = false;
	    }
	}
	term->need_positions();
	terms.push_back(term);
    }

    void adjust_window(size_t alternative_window) {
	if (alternative_window > window) window = alternative_window;
    }

    /// Convert to a Xapian::Query * using adjacent OP_PHRASE.
    Query * as_phrase_query() const {
	return as_opwindow_query(Query::OP_PHRASE, 0);
    }

    /// Convert to a Xapian::Query * using adjacent OP_PHRASE.
    Query* as_synonym_phrase_query(State* state) const {
	string name;
	termpos pos = 0;
	for (Term* t : terms) {
	    if (!name.empty()) {
		name += ' ';
	    } else {
		pos = t->pos;
	    }
	    name += t->name;
	}

	for (auto&& prefix : *prefixes) {
	    // Only try unstemmed for multi-word.
	    string term;
	    if (!prefix.empty()) {
		term += prefix;
		if (prefix_needs_colon(prefix, name[0])) term += ':';
	    }
	    term += name;

	    Xapian::Database db = state->get_database();
	    Xapian::TermIterator syn = db.synonyms_begin(term);
	    Xapian::TermIterator end = db.synonyms_end(term);

	    // Caution: this does `delete this;`!
	    Query* q = as_opwindow_query(Query::OP_PHRASE, 0);
	    // FIXME: Is this right when there's more than one entry in
	    // prefixes?
	    Query* q2 = new Query(q->OP_SYNONYM,
				  SynonymIterator(syn, pos, q),
				  SynonymIterator(end));
	    delete q;
	    return q2;
	    // FIXME: Handle multiple prefixes properly...
	}
	return new Query();
    }

    /// Convert to a Xapian::Query * using OP_NEAR.
    Query * as_near_query() const {
	// The common meaning of 'a NEAR b' is "a within 10 terms of b", which
	// means a window size of 11.  For more than 2 terms, we just add one
	// to the window size for each extra term.
	size_t w = window;
	if (w == 0) w = 10;
	return as_opwindow_query(Query::OP_NEAR, w - 1);
    }

    /// Convert to a Xapian::Query * using OP_PHRASE to implement ADJ.
    Query * as_adj_query() const {
	// The common meaning of 'a ADJ b' is "a at most 10 terms before b",
	// which means a window size of 11.  For more than 2 terms, we just add
	// one to the window size for each extra term.
	size_t w = window;
	if (w == 0) w = 10;
	return as_opwindow_query(Query::OP_PHRASE, w - 1);
    }
};

void
Term::as_positional_unbroken(Terms* terms) const
{
#ifdef USE_ICU
    if (state->flags & QueryParser::FLAG_WORD_BREAKS) {
	for (WordIterator tk(name); tk != WordIterator(); ++tk) {
	    const string& t = *tk;
	    Term * c = new Term(state, t, field_info, unstemmed, stem, pos);
	    terms->add_positional_term(c);
	}
	delete this;
	return;
    }
#endif
    // Add each individual character to the phrase.
    string t;
    for (Utf8Iterator it(name); it != Utf8Iterator(); ++it) {
	Unicode::append_utf8(t, *it);
	Term * c = new Term(state, t, field_info, unstemmed, stem, pos);
	terms->add_positional_term(c);
	t.resize(0);
    }

    // FIXME: we want to add the n-grams as filters too for efficiency.

    delete this;
}

// Helper macro to check for missing arguments to a boolean operator.
#define VET_BOOL_ARGS(A, B, OP_TXT) \
    do {\
	if (!A || !B) {\
	    state->error = "Syntax: <expression> " OP_TXT " <expression>";\
	    yy_parse_failed(yypParser);\
	    return;\
	}\
    } while (0)

#line 1560 "queryparser/queryparser_internal.cc"
/**************** End of %include directives **********************************/
/* These constants specify the various numeric values for terminal symbols
** in a format understandable to "makeheaders".  This section is blank unless
** "lemon" is run with the "-m" command-line option.
***************** Begin makeheaders token definitions *************************/
/**************** End makeheaders token definitions ***************************/

/* The next section is a series of control #defines.
** various aspects of the generated parser.
**    YYCODETYPE         is the data type used to store the integer codes
**                       that represent terminal and non-terminal symbols.
**                       "unsigned char" is used if there are fewer than
**                       256 symbols.  Larger types otherwise.
**    YYNOCODE           is a number of type YYCODETYPE that is not used for
**                       any terminal or nonterminal symbol.
**    YYFALLBACK         If defined, this indicates that one or more tokens
**                       (also known as: "terminal symbols") have fall-back
**                       values which should be used if the original symbol
**                       would not parse.  This permits keywords to sometimes
**                       be used as identifiers, for example.
**    YYACTIONTYPE       is the data type used for "action codes" - numbers
**                       that indicate what to do in response to the next
**                       token.
**    ParseTOKENTYPE     is the data type used for minor type for terminal
**                       symbols.  Background: A "minor type" is a semantic
**                       value associated with a terminal or non-terminal
**                       symbols.  For example, for an "ID" terminal symbol,
**                       the minor type might be the name of the identifier.
**                       Each non-terminal can have a different minor type.
**                       Terminal symbols all have the same minor type, though.
**                       This macros defines the minor type for terminal 
**                       symbols.
**    YYMINORTYPE        is the data type used for all minor types.
**                       This is typically a union of many types, one of
**                       which is ParseTOKENTYPE.  The entry in the union
**                       for terminal symbols is called "yy0".
**    YYSTACKDEPTH       is the maximum depth of the parser's stack.  If
**                       zero the stack is dynamically sized using realloc()
**    ParseARG_SDECL     A static variable declaration for the %extra_argument
**    ParseARG_PDECL     A parameter declaration for the %extra_argument
**    ParseARG_STORE     Code to store %extra_argument into yypParser
**    ParseARG_FETCH     Code to extract %extra_argument from yypParser
**    YYERRORSYMBOL      is the code number of the error symbol.  If not
**                       defined, then do no error processing.
**    YYNSTATE           the combined number of states.
**    YYNRULE            the number of rules in the grammar
**    YYNTOKEN           Number of terminal symbols
**    YY_MAX_SHIFT       Maximum value for shift actions
**    YY_MIN_SHIFTREDUCE Minimum value for shift-reduce actions
**    YY_MAX_SHIFTREDUCE Maximum value for shift-reduce actions
**    YY_ERROR_ACTION    The yy_action[] code for syntax error
**    YY_ACCEPT_ACTION   The yy_action[] code for accept
**    YY_NO_ACTION       The yy_action[] code for no-op
**    YY_MIN_REDUCE      Minimum value for reduce actions
**    YY_MAX_REDUCE      Maximum value for reduce actions
*/
#ifndef INTERFACE
# define INTERFACE 1
#endif
/************* Begin control #defines *****************************************/
#define YYCODETYPE unsigned char
#define YYNOCODE 42
#define YYACTIONTYPE unsigned char
#define ParseTOKENTYPE Term *
typedef union {
  int yyinit;
  ParseTOKENTYPE yy0;
  Query * yy1;
  ProbQuery * yy18;
  TermGroup * yy32;
  int yy34;
  Terms * yy36;
} YYMINORTYPE;
#ifndef YYSTACKDEPTH
#define YYSTACKDEPTH 100
#endif
#define ParseARG_SDECL State * state;
#define ParseARG_PDECL ,State * state
#define ParseARG_FETCH State * state = yypParser->state
#define ParseARG_STORE yypParser->state = state
#define YYNSTATE             42
#define YYNRULE              59
#define YYNTOKEN             26
#define YY_MAX_SHIFT         41
#define YY_MIN_SHIFTREDUCE   83
#define YY_MAX_SHIFTREDUCE   141
#define YY_ERROR_ACTION      142
#define YY_ACCEPT_ACTION     143
#define YY_NO_ACTION         144
#define YY_MIN_REDUCE        145
#define YY_MAX_REDUCE        203
/************* End control #defines *******************************************/

/* Define the yytestcase() macro to be a no-op if is not already defined
** otherwise.
**
** Applications can choose to define yytestcase() in the %include section
** to a macro that can assist in verifying code coverage.  For production
** code the yytestcase() macro should be turned off.  But it is useful
** for testing.
*/
#ifndef yytestcase
# define yytestcase(X)
#endif


/* Next are the tables used to determine what action to take based on the
** current state and lookahead token.  These tables are used to implement
** functions that take a state number and lookahead value and return an
** action integer.  
**
** Suppose the action integer is N.  Then the action is determined as
** follows
**
**   0 <= N <= YY_MAX_SHIFT             Shift N.  That is, push the lookahead
**                                      token onto the stack and goto state N.
**
**   N between YY_MIN_SHIFTREDUCE       Shift to an arbitrary state then
**     and YY_MAX_SHIFTREDUCE           reduce by rule N-YY_MIN_SHIFTREDUCE.
**
**   N == YY_ERROR_ACTION               A syntax error has occurred.
**
**   N == YY_ACCEPT_ACTION              The parser accepts its input.
**
**   N == YY_NO_ACTION                  No such action.  Denotes unused
**                                      slots in the yy_action[] table.
**
**   N between YY_MIN_REDUCE            Reduce by rule N-YY_MIN_REDUCE
**     and YY_MAX_REDUCE
**
** The action table is constructed as a single large table named yy_action[].
** Given state S and lookahead X, the action is computed as either:
**
**    (A)   N = yy_action[ yy_shift_ofst[S] + X ]
**    (B)   N = yy_default[S]
**
** The (A) formula is preferred.  The B formula is used instead if
** yy_lookahead[yy_shift_ofst[S]+X] is not equal to X.
**
** The formulas above are for computing the action when the lookahead is
** a terminal symbol.  If the lookahead is a non-terminal (as occurs after
** a reduce action) then the yy_reduce_ofst[] array is used in place of
** the yy_shift_ofst[] array.
**
** The following are the tables generated in this section:
**
**  yy_action[]        A single table containing all actions.
**  yy_lookahead[]     A table containing the lookahead for each entry in
**                     yy_action.  Used to detect hash collisions.
**  yy_shift_ofst[]    For each state, the offset into yy_action for
**                     shifting terminals.
**  yy_reduce_ofst[]   For each state, the offset into yy_action for
**                     shifting non-terminals after a reduce.
**  yy_default[]       Default action for each state.
**
*********** Begin parsing tables **********************************************/
#define YY_ACTTAB_COUNT (352)
static const YYACTIONTYPE yy_action[] = {
 /*     0 */    24,   25,  145,  144,  144,    3,  144,   34,   11,   10,
 /*    10 */     2,   27,  144,   17,   13,   12,  111,  112,  113,  104,
 /*    20 */    94,   16,    4,  146,  122,  105,   95,    7,    6,    1,
 /*    30 */     8,   11,   10,  119,   27,  123,   17,    5,    5,  111,
 /*    40 */   112,  113,  104,   94,   16,    4,  124,  122,  143,   41,
 /*    50 */    41,   19,    9,   41,   21,   14,   18,  135,   36,   28,
 /*    60 */    35,   33,   32,   40,   40,   40,    9,   40,   21,   14,
 /*    70 */    18,  130,   36,   28,   35,   33,   11,   10,  128,   27,
 /*    80 */   133,   17,  131,  120,  111,  112,  113,  104,   94,   16,
 /*    90 */     4,   15,  122,   29,   29,   29,    9,   29,   21,   14,
 /*   100 */    18,  134,   36,   28,   35,   33,   30,   30,   30,    9,
 /*   110 */    30,   21,   14,   18,  132,   36,   28,   35,   33,   31,
 /*   120 */    31,   19,    9,   31,   21,   14,   18,  144,   36,   28,
 /*   130 */    35,   33,  153,  153,  153,    9,  153,   21,   14,   18,
 /*   140 */   144,   36,   28,   35,   33,   26,   26,   26,    9,   26,
 /*   150 */    21,   14,   18,  144,   36,   28,   35,   33,   23,   23,
 /*   160 */    23,    9,   23,   21,   14,   18,  144,   36,   28,   35,
 /*   170 */    33,   39,   39,   39,    9,   39,   21,   14,   18,  144,
 /*   180 */    36,   28,   35,   33,  201,  201,  144,   27,  144,   22,
 /*   190 */   144,  144,  111,  112,  113,  201,  201,   16,    4,  172,
 /*   200 */   122,  172,  172,  172,  172,   38,   37,   38,   37,    1,
 /*   210 */     8,  172,  144,  129,  127,  129,  127,    5,   27,  144,
 /*   220 */    20,  144,  172,  111,  112,  113,  102,  144,   16,    4,
 /*   230 */    27,  122,   20,  144,  144,  111,  112,  113,  106,  144,
 /*   240 */    16,    4,   27,  122,   20,  144,  144,  111,  112,  113,
 /*   250 */   103,  144,   16,    4,   27,  122,   20,  144,  144,  111,
 /*   260 */   112,  113,  107,  144,   16,    4,   27,  122,   22,  144,
 /*   270 */   144,  111,  112,  113,  144,  144,   16,    4,  203,  122,
 /*   280 */   203,  203,  203,  203,  144,  144,  144,  144,  159,  159,
 /*   290 */   203,   36,   28,   35,   33,  144,  144,  162,  144,  144,
 /*   300 */   162,  203,   36,   28,   35,   33,  160,  144,  125,  160,
 /*   310 */   144,   36,   28,   35,   33,  163,  121,  125,  163,  126,
 /*   320 */    36,   28,   35,   33,  161,  114,  144,  161,  126,   36,
 /*   330 */    28,   35,   33,  144,  144,  158,  158,  144,   36,   28,
 /*   340 */    35,   33,    6,    1,    8,  144,  144,  144,  144,  144,
 /*   350 */   144,    5,
};
static const YYCODETYPE yy_lookahead[] = {
 /*     0 */    36,   36,    0,   41,   41,    5,   41,    6,    8,    9,
 /*    10 */    10,   11,   41,   13,    8,    9,   16,   17,   18,   19,
 /*    20 */    20,   21,   22,    0,   24,   19,   20,    2,    3,    4,
 /*    30 */     5,    8,    9,   23,   11,   13,   13,   12,   12,   16,
 /*    40 */    17,   18,   19,   20,   21,   22,   24,   24,   27,   28,
 /*    50 */    29,   30,   31,   32,   33,   34,   35,   13,   37,   38,
 /*    60 */    39,   40,    7,   28,   29,   30,   31,   32,   33,   34,
 /*    70 */    35,   14,   37,   38,   39,   40,    8,    9,   15,   11,
 /*    80 */    13,   13,   25,   13,   16,   17,   18,   19,   20,   21,
 /*    90 */    22,   21,   24,   28,   29,   30,   31,   32,   33,   34,
 /*   100 */    35,   13,   37,   38,   39,   40,   28,   29,   30,   31,
 /*   110 */    32,   33,   34,   35,   13,   37,   38,   39,   40,   28,
 /*   120 */    29,   30,   31,   32,   33,   34,   35,   41,   37,   38,
 /*   130 */    39,   40,   28,   29,   30,   31,   32,   33,   34,   35,
 /*   140 */    41,   37,   38,   39,   40,   28,   29,   30,   31,   32,
 /*   150 */    33,   34,   35,   41,   37,   38,   39,   40,   28,   29,
 /*   160 */    30,   31,   32,   33,   34,   35,   41,   37,   38,   39,
 /*   170 */    40,   28,   29,   30,   31,   32,   33,   34,   35,   41,
 /*   180 */    37,   38,   39,   40,    8,    9,   41,   11,   41,   13,
 /*   190 */    41,   41,   16,   17,   18,   19,   20,   21,   22,    0,
 /*   200 */    24,    2,    3,    4,    5,    6,    7,    6,    7,    4,
 /*   210 */     5,   12,   41,   14,   15,   14,   15,   12,   11,   41,
 /*   220 */    13,   41,   23,   16,   17,   18,   19,   41,   21,   22,
 /*   230 */    11,   24,   13,   41,   41,   16,   17,   18,   19,   41,
 /*   240 */    21,   22,   11,   24,   13,   41,   41,   16,   17,   18,
 /*   250 */    19,   41,   21,   22,   11,   24,   13,   41,   41,   16,
 /*   260 */    17,   18,   19,   41,   21,   22,   11,   24,   13,   41,
 /*   270 */    41,   16,   17,   18,   41,   41,   21,   22,    0,   24,
 /*   280 */     2,    3,    4,    5,   41,   41,   41,   41,   34,   35,
 /*   290 */    12,   37,   38,   39,   40,   41,   41,   32,   41,   41,
 /*   300 */    35,   23,   37,   38,   39,   40,   32,   41,   13,   35,
 /*   310 */    41,   37,   38,   39,   40,   32,   21,   13,   35,   24,
 /*   320 */    37,   38,   39,   40,   32,   21,   41,   35,   24,   37,
 /*   330 */    38,   39,   40,   41,   41,   34,   35,   41,   37,   38,
 /*   340 */    39,   40,    3,    4,    5,   41,   41,   41,   41,   41,
 /*   350 */    41,   12,   41,   41,   41,   41,   41,   41,   41,   41,
 /*   360 */    41,   41,   41,   41,   41,   41,   41,   41,   41,
};
#define YY_SHIFT_COUNT    (41)
#define YY_SHIFT_MIN      (0)
#define YY_SHIFT_MAX      (339)
static const unsigned short int yy_shift_ofst[] = {
 /*     0 */    23,    0,   68,   68,   68,   68,   68,   68,   68,  176,
 /*    10 */   207,  219,  231,  243,  255,   22,   22,  199,  278,   25,
 /*    20 */   201,    6,  201,  339,  295,  304,  205,   70,   57,   26,
 /*    30 */    26,   10,   44,   55,   67,    1,   63,   88,  101,   26,
 /*    40 */    26,    2,
};
#define YY_REDUCE_COUNT (16)
#define YY_REDUCE_MIN   (-36)
#define YY_REDUCE_MAX   (301)
static const short yy_reduce_ofst[] = {
 /*     0 */    21,   35,   65,   78,   91,  104,  117,  130,  143,  254,
 /*    10 */   265,  274,  283,  292,  301,  -36,  -35,
};
static const YYACTIONTYPE yy_default[] = {
 /*     0 */   154,  154,  154,  154,  154,  154,  154,  154,  154,  155,
 /*    10 */   142,  142,  142,  142,  170,  142,  142,  171,  202,  142,
 /*    20 */   172,  142,  171,  151,  142,  142,  152,  142,  178,  150,
 /*    30 */   149,  199,  142,  180,  142,  179,  177,  142,  142,  148,
 /*    40 */   147,  199,
};
/********** End of lemon-generated parsing tables *****************************/

/* The next table maps tokens (terminal symbols) into fallback tokens.  
** If a construct like the following:
** 
**      %fallback ID X Y Z.
**
** appears in the grammar, then ID becomes a fallback token for X, Y,
** and Z.  Whenever one of the tokens X, Y, or Z is input to the parser
** but it does not parse, the type of the token is changed to ID and
** the parse is retried before an error is thrown.
**
** This feature can be used, for example, to cause some keywords in a language
** to revert to identifiers if they keyword does not apply in the context where
** it appears.
*/
#ifdef YYFALLBACK
static const YYCODETYPE yyFallback[] = {
};
#endif /* YYFALLBACK */

/* The following structure represents a single element of the
** parser's stack.  Information stored includes:
**
**   +  The state number for the parser at this level of the stack.
**
**   +  The value of the token stored at this level of the stack.
**      (In other words, the "major" token.)
**
**   +  The semantic value stored at this level of the stack.  This is
**      the information used by the action routines in the grammar.
**      It is sometimes called the "minor" token.
**
** After the "shift" half of a SHIFTREDUCE action, the stateno field
** actually contains the reduce action for the second half of the
** SHIFTREDUCE.
*/
struct yyStackEntry {
  yyStackEntry() {
    stateno = 0;
    major = 0;
  }
  yyStackEntry(YYACTIONTYPE stateno_, YYCODETYPE major_, ParseTOKENTYPE minor_) {
    stateno = stateno_;
    major = major_;
    minor.yy0 = minor_;
  }
  YYACTIONTYPE stateno;  /* The state-number, or reduce action in SHIFTREDUCE */
  YYCODETYPE major;      /* The major token value.  This is the code
                         ** number for the token at this stack level */
  YYMINORTYPE minor;     /* The user-supplied minor token value.  This
                         ** is the value of the token  */
};

static void ParseInit(yyParser *pParser);
static void ParseFinalize(yyParser *pParser);

/* The state of the parser is completely contained in an instance of
** the following structure */
struct yyParser {
#ifdef YYTRACKMAXSTACKDEPTH
  int yyhwm;                    /* High-water mark of the stack */
#endif
#ifndef YYNOERRORRECOVERY
  int yyerrcnt;                 /* Shifts left before out of the error */
#endif
  ParseARG_SDECL                /* A place to hold %extra_argument */
  vector<yyStackEntry> yystack; /* The parser's stack */
  yyParser() {
    ParseInit(this);
  }
  ~yyParser() {
    ParseFinalize(this);
  }
};
typedef struct yyParser yyParser;

#include "omassert.h"
#include "debuglog.h"

#if defined(YYCOVERAGE) || defined(XAPIAN_DEBUG_LOG)
/* For tracing shifts, the names of all terminals and nonterminals
** are required.  The following table supplies these names */
static const char *const yyTokenName[] = { 
  /*    0 */ "$",
  /*    1 */ "ERROR",
  /*    2 */ "OR",
  /*    3 */ "XOR",
  /*    4 */ "AND",
  /*    5 */ "NOT",
  /*    6 */ "NEAR",
  /*    7 */ "ADJ",
  /*    8 */ "LOVE",
  /*    9 */ "HATE",
  /*   10 */ "HATE_AFTER_AND",
  /*   11 */ "SYNONYM",
  /*   12 */ "SYN",
  /*   13 */ "TERM",
  /*   14 */ "GROUP_TERM",
  /*   15 */ "PHR_TERM",
  /*   16 */ "EDIT_TERM",
  /*   17 */ "WILD_TERM",
  /*   18 */ "PARTIAL_TERM",
  /*   19 */ "BOOLEAN_FILTER",
  /*   20 */ "RANGE",
  /*   21 */ "QUOTE",
  /*   22 */ "BRA",
  /*   23 */ "KET",
  /*   24 */ "UNBROKEN_WORDS",
  /*   25 */ "EMPTY_GROUP_OK",
  /*   26 */ "error",
  /*   27 */ "query",
  /*   28 */ "expr",
  /*   29 */ "prob_expr",
  /*   30 */ "bool_arg",
  /*   31 */ "prob",
  /*   32 */ "term",
  /*   33 */ "stop_prob",
  /*   34 */ "stop_term",
  /*   35 */ "compound_term",
  /*   36 */ "phrase",
  /*   37 */ "phrased_term",
  /*   38 */ "group",
  /*   39 */ "near_expr",
  /*   40 */ "adj_expr",
};

/* For tracing reduce actions, the names of all rules are required.
*/
static const char *const yyRuleName[] = {
 /*   0 */ "query ::= expr",
 /*   1 */ "query ::=",
 /*   2 */ "expr ::= bool_arg AND bool_arg",
 /*   3 */ "expr ::= bool_arg NOT bool_arg",
 /*   4 */ "expr ::= bool_arg AND NOT bool_arg",
 /*   5 */ "expr ::= bool_arg AND HATE_AFTER_AND bool_arg",
 /*   6 */ "expr ::= bool_arg OR bool_arg",
 /*   7 */ "expr ::= bool_arg XOR bool_arg",
 /*   8 */ "expr ::= bool_arg SYN bool_arg",
 /*   9 */ "bool_arg ::=",
 /*  10 */ "prob_expr ::= prob",
 /*  11 */ "prob ::= RANGE",
 /*  12 */ "prob ::= stop_prob RANGE",
 /*  13 */ "prob ::= stop_term stop_term",
 /*  14 */ "prob ::= prob stop_term",
 /*  15 */ "prob ::= LOVE term",
 /*  16 */ "prob ::= stop_prob LOVE term",
 /*  17 */ "prob ::= HATE term",
 /*  18 */ "prob ::= stop_prob HATE term",
 /*  19 */ "prob ::= HATE BOOLEAN_FILTER",
 /*  20 */ "prob ::= stop_prob HATE BOOLEAN_FILTER",
 /*  21 */ "prob ::= BOOLEAN_FILTER",
 /*  22 */ "prob ::= stop_prob BOOLEAN_FILTER",
 /*  23 */ "prob ::= LOVE BOOLEAN_FILTER",
 /*  24 */ "prob ::= stop_prob LOVE BOOLEAN_FILTER",
 /*  25 */ "stop_prob ::= stop_term",
 /*  26 */ "stop_term ::= TERM",
 /*  27 */ "term ::= TERM",
 /*  28 */ "compound_term ::= EDIT_TERM",
 /*  29 */ "compound_term ::= WILD_TERM",
 /*  30 */ "compound_term ::= PARTIAL_TERM",
 /*  31 */ "compound_term ::= QUOTE phrase QUOTE",
 /*  32 */ "compound_term ::= phrased_term",
 /*  33 */ "compound_term ::= group",
 /*  34 */ "compound_term ::= near_expr",
 /*  35 */ "compound_term ::= adj_expr",
 /*  36 */ "compound_term ::= BRA expr KET",
 /*  37 */ "compound_term ::= SYNONYM TERM",
 /*  38 */ "compound_term ::= SYNONYM QUOTE phrase QUOTE",
 /*  39 */ "compound_term ::= UNBROKEN_WORDS",
 /*  40 */ "phrase ::= TERM",
 /*  41 */ "phrase ::= UNBROKEN_WORDS",
 /*  42 */ "phrase ::= phrase TERM",
 /*  43 */ "phrase ::= phrase UNBROKEN_WORDS",
 /*  44 */ "phrased_term ::= TERM PHR_TERM",
 /*  45 */ "phrased_term ::= phrased_term PHR_TERM",
 /*  46 */ "group ::= TERM GROUP_TERM",
 /*  47 */ "group ::= group GROUP_TERM",
 /*  48 */ "group ::= group EMPTY_GROUP_OK",
 /*  49 */ "near_expr ::= TERM NEAR TERM",
 /*  50 */ "near_expr ::= near_expr NEAR TERM",
 /*  51 */ "adj_expr ::= TERM ADJ TERM",
 /*  52 */ "adj_expr ::= adj_expr ADJ TERM",
 /*  53 */ "expr ::= prob_expr",
 /*  54 */ "bool_arg ::= expr",
 /*  55 */ "prob_expr ::= term",
 /*  56 */ "stop_prob ::= prob",
 /*  57 */ "stop_term ::= compound_term",
 /*  58 */ "term ::= compound_term",
};

/*
** This function returns the symbolic name associated with a token
** value.
*/
static const char *ParseTokenName(int tokenType){
  if( tokenType>=0 && tokenType<(int)(sizeof(yyTokenName)/sizeof(yyTokenName[0])) ){
    return yyTokenName[tokenType];
  }
  return "Unknown";
}

/*
** This function returns the symbolic name associated with a rule
** value.
*/
static const char *ParseRuleName(int ruleNum){
  if( ruleNum>=0 && ruleNum<(int)(sizeof(yyRuleName)/sizeof(yyRuleName[0])) ){
    return yyRuleName[ruleNum];
  }
  return "Unknown";
}
#endif /* defined(YYCOVERAGE) || defined(XAPIAN_DEBUG_LOG) */

/* Datatype of the argument to the memory allocated passed as the
** second argument to ParseAlloc() below.  This can be changed by
** putting an appropriate #define in the %include section of the input
** grammar.
*/
#ifndef YYMALLOCARGTYPE
# define YYMALLOCARGTYPE size_t
#endif

/* Initialize a new parser that has already been allocated.
*/
static
void ParseInit(yyParser *pParser){
#ifdef YYTRACKMAXSTACKDEPTH
  pParser->yyhwm = 0;
#endif
#if 0
#if YYSTACKDEPTH<=0
  pParser->yytos = NULL;
  pParser->yystack = NULL;
  pParser->yystksz = 0;
  if( yyGrowStack(pParser) ){
    pParser->yystack = &pParser->yystk0;
    pParser->yystksz = 1;
  }
#endif
#endif
#ifndef YYNOERRORRECOVERY
  pParser->yyerrcnt = -1;
#endif
#if 0
  pParser->yytos = pParser->yystack;
  pParser->yystack[0].stateno = 0;
  pParser->yystack[0].major = 0;
#if YYSTACKDEPTH>0
  pParser->yystackEnd = &pParser->yystack[YYSTACKDEPTH-1];
#endif
#else
  pParser->yystack.push_back(yyStackEntry());
#endif
}

#ifndef Parse_ENGINEALWAYSONSTACK
/* 
** This function allocates a new parser.
**
** Inputs:
** None.
**
** Outputs:
** A pointer to a parser.  This pointer is used in subsequent calls
** to Parse and ParseFree.
*/
static yyParser *ParseAlloc(void){
  return new yyParser;
}
#endif /* Parse_ENGINEALWAYSONSTACK */


/* The following function deletes the "minor type" or semantic value
** associated with a symbol.  The symbol can be either a terminal
** or nonterminal. "yymajor" is the symbol code, and "yypminor" is
** a pointer to the value to be deleted.  The code used to do the 
** deletions is derived from the %destructor and/or %token_destructor
** directives of the input grammar.
*/
static void yy_destructor(
  yyParser *yypParser,    /* The parser */
  YYCODETYPE yymajor,     /* Type code for object to destroy */
  YYMINORTYPE *yypminor   /* The object to be destroyed */
){
  ParseARG_FETCH;
  switch( yymajor ){
    /* Here is inserted the actions which take place when a
    ** terminal or non-terminal is destroyed.  This can happen
    ** when the symbol is popped from the stack during a
    ** reduce or during error processing or when a parser is 
    ** being destroyed before it is finished parsing.
    **
    ** Note: during a reduce, the only symbols destroyed are those
    ** which appear on the RHS of the rule, but which are *not* used
    ** inside the C code.
    */
/********* Begin destructor definitions ***************************************/
      /* TERMINAL Destructor */
    case 1: /* ERROR */
    case 2: /* OR */
    case 3: /* XOR */
    case 4: /* AND */
    case 5: /* NOT */
    case 6: /* NEAR */
    case 7: /* ADJ */
    case 8: /* LOVE */
    case 9: /* HATE */
    case 10: /* HATE_AFTER_AND */
    case 11: /* SYNONYM */
    case 12: /* SYN */
    case 13: /* TERM */
    case 14: /* GROUP_TERM */
    case 15: /* PHR_TERM */
    case 16: /* EDIT_TERM */
    case 17: /* WILD_TERM */
    case 18: /* PARTIAL_TERM */
    case 19: /* BOOLEAN_FILTER */
    case 20: /* RANGE */
    case 21: /* QUOTE */
    case 22: /* BRA */
    case 23: /* KET */
    case 24: /* UNBROKEN_WORDS */
    case 25: /* EMPTY_GROUP_OK */
{
#line 2218 "queryparser/queryparser.lemony"
 delete (yypminor->yy0); 
#line 2146 "queryparser/queryparser_internal.cc"
}
      break;
    case 28: /* expr */
    case 29: /* prob_expr */
    case 30: /* bool_arg */
    case 32: /* term */
    case 34: /* stop_term */
    case 35: /* compound_term */
{
#line 2296 "queryparser/queryparser.lemony"
 delete (yypminor->yy1); 
#line 2158 "queryparser/queryparser_internal.cc"
}
      break;
    case 31: /* prob */
    case 33: /* stop_prob */
{
#line 2416 "queryparser/queryparser.lemony"
 delete (yypminor->yy18); 
#line 2166 "queryparser/queryparser_internal.cc"
}
      break;
    case 36: /* phrase */
    case 37: /* phrased_term */
    case 39: /* near_expr */
    case 40: /* adj_expr */
{
#line 2613 "queryparser/queryparser.lemony"
 delete (yypminor->yy36); 
#line 2176 "queryparser/queryparser_internal.cc"
}
      break;
    case 38: /* group */
{
#line 2654 "queryparser/queryparser.lemony"
 delete (yypminor->yy32); 
#line 2183 "queryparser/queryparser_internal.cc"
}
      break;
/********* End destructor definitions *****************************************/
    default:  break;   /* If no destructor action specified: do nothing */
  }
  ParseARG_STORE; /* Suppress warning about unused %extra_argument variable */
}

/*
** Pop the parser's stack once.
**
** If there is a destructor routine associated with the token which
** is popped from the stack, then call it.
*/
static void yy_pop_parser_stack(yyParser *pParser){
  Assert( pParser->yystack.size() > 1 );
  yyStackEntry *yytos = &pParser->yystack.back();

  LOGLINE(QUERYPARSER, "Popping " << ParseTokenName(yytos->major));
  yy_destructor(pParser, yytos->major, &yytos->minor);
  pParser->yystack.pop_back();
}

/*
** Clear all secondary memory allocations from the parser
*/
static
void ParseFinalize(yyParser *pParser){
  while( pParser->yystack.size() > 1 ) yy_pop_parser_stack(pParser);
}

#ifndef Parse_ENGINEALWAYSONSTACK
/* 
** Deallocate and destroy a parser.  Destructors are called for
** all stack elements before shutting the parser down.
**
** If the YYPARSEFREENEVERNULL macro exists (for example because it
** is defined in a %include section of the input grammar) then it is
** assumed that the input pointer is never NULL.
*/
static
void ParseFree(
  yyParser *pParser           /* The parser to be deleted */
){
  delete pParser;
}
#endif /* Parse_ENGINEALWAYSONSTACK */

/*
** Return the peak depth of the stack for a parser.
*/
#ifdef YYTRACKMAXSTACKDEPTH
int ParseStackPeak(yyParser *pParser){
  return pParser->yyhwm;
}
#endif

/* This array of booleans keeps track of the parser statement
** coverage.  The element yycoverage[X][Y] is set when the parser
** is in state X and has a lookahead token Y.  In a well-tested
** systems, every element of this matrix should end up being set.
*/
#if defined(YYCOVERAGE)
static unsigned char yycoverage[YYNSTATE][YYNTOKEN];
#endif

/*
** Write into out a description of every state/lookahead combination that
**
**   (1)  has not been used by the parser, and
**   (2)  is not a syntax error.
**
** Return the number of missed state/lookahead combinations.
*/
#if defined(YYCOVERAGE)
int ParseCoverage(FILE *out){
  int stateno, iLookAhead, i;
  int nMissed = 0;
  for(stateno=0; stateno<YYNSTATE; stateno++){
    i = yy_shift_ofst[stateno];
    for(iLookAhead=0; iLookAhead<YYNTOKEN; iLookAhead++){
      if( yy_lookahead[i+iLookAhead]!=iLookAhead ) continue;
      if( yycoverage[stateno][iLookAhead]==0 ) nMissed++;
      if( out ){
        fprintf(out,"State %d lookahead %s %s\n", stateno,
                yyTokenName[iLookAhead],
                yycoverage[stateno][iLookAhead] ? "ok" : "missed");
      }
    }
  }
  return nMissed;
}
#endif

/*
** Find the appropriate action for a parser given the terminal
** look-ahead token iLookAhead.
*/
static unsigned int yy_find_shift_action(
  yyParser *pParser,        /* The parser */
  YYCODETYPE iLookAhead     /* The look-ahead token */
){
  int i;
  int stateno = pParser->yystack.back().stateno;
 
  if( stateno>YY_MAX_SHIFT ) return stateno;
  Assert( stateno <= YY_SHIFT_COUNT );
#if defined(YYCOVERAGE)
  yycoverage[stateno][iLookAhead] = 1;
#endif
  do{
    i = yy_shift_ofst[stateno];
    Assert( i>=0 );
    Assert( i+YYNTOKEN<=(int)(sizeof(yy_lookahead)/sizeof(yy_lookahead[0])) );
    Assert( iLookAhead!=YYNOCODE );
    Assert( iLookAhead < YYNTOKEN );
    i += iLookAhead;
    if( yy_lookahead[i]!=iLookAhead ){
#ifdef YYFALLBACK
      YYCODETYPE iFallback;            /* Fallback token */
      if( iLookAhead<sizeof(yyFallback)/sizeof(yyFallback[0])
             && (iFallback = yyFallback[iLookAhead])!=0 ){
        LOGLINE(QUERYPARSER,
                "FALLBACK " << ParseTokenName(iLookAhead) << " => " <<
                ParseTokenName(iFallback));
        Assert( yyFallback[iFallback]==0 ); /* Fallback loop must terminate */
        iLookAhead = iFallback;
        continue;
      }
#endif
#ifdef YYWILDCARD
      {
        int j = i - iLookAhead + YYWILDCARD;
        if( 
#if YY_SHIFT_MIN+YYWILDCARD<0
          j>=0 &&
#endif
#if YY_SHIFT_MAX+YYWILDCARD>=YY_ACTTAB_COUNT
          j<YY_ACTTAB_COUNT &&
#endif
          yy_lookahead[j]==YYWILDCARD && iLookAhead>0
        ){
          LOGLINE(QUERYPARSER,
                  "WILDCARD " << ParseTokenName(iLookAhead) << " => " <<
                  ParseTokenName(YYWILDCARD));
          return yy_action[j];
        }
      }
#endif /* YYWILDCARD */
      return yy_default[stateno];
    }else{
      return yy_action[i];
    }
  }while(1);
}

/*
** Find the appropriate action for a parser given the non-terminal
** look-ahead token iLookAhead.
*/
static int yy_find_reduce_action(
  int stateno,              /* Current state number */
  YYCODETYPE iLookAhead     /* The look-ahead token */
){
  int i;
#ifdef YYERRORSYMBOL
  if( stateno>YY_REDUCE_COUNT ){
    return yy_default[stateno];
  }
#else
  Assert( stateno<=YY_REDUCE_COUNT );
#endif
  i = yy_reduce_ofst[stateno];
  Assert( iLookAhead!=YYNOCODE );
  i += iLookAhead;
#ifdef YYERRORSYMBOL
  if( i<0 || i>=YY_ACTTAB_COUNT || yy_lookahead[i]!=iLookAhead ){
    return yy_default[stateno];
  }
#else
  Assert( i>=0 && i<YY_ACTTAB_COUNT );
  Assert( yy_lookahead[i]==iLookAhead );
#endif
  return yy_action[i];
}

/*
** The following routine is called if the stack overflows.
** In Xapian this can never happen as we use std::vector to provide a stack
** of indefinite size.
*/
#if 0
static void yyStackOverflow(yyParser *yypParser){
   ParseARG_FETCH;
   yypParser->yyidx--;
#ifndef NDEBUG
   if( yyTraceFILE ){
     fprintf(yyTraceFILE,"%sStack Overflow!\n",yyTracePrompt);
   }
#endif
   while( yypParser->yytos>yypParser->yystack ) yy_pop_parser_stack(yypParser);
   /* Here code is inserted which will execute if the parser
   ** stack ever overflows */
/******** Begin %stack_overflow code ******************************************/
/******** End %stack_overflow code ********************************************/
   ParseARG_STORE; /* Suppress warning about unused %extra_argument var */
}
#endif

/*
** Print tracing information for a SHIFT action
*/
#ifdef XAPIAN_DEBUG_LOG
static void yyTraceShift(yyParser *yypParser, int yyNewState, const char *zTag){
  if( yyNewState<YYNSTATE ){
    LOGLINE(QUERYPARSER, zTag << " '" <<
                         yyTokenName[yypParser->yystack.back().major] <<
                         "', go to state " << yyNewState);
  }else{
    LOGLINE(QUERYPARSER, zTag << " '" <<
                         yyTokenName[yypParser->yystack.back().major] <<
                         "', pending reduce " << yyNewState - YY_MIN_REDUCE);
  }
}
#else
# define yyTraceShift(X,Y,Z)
#endif

/*
** Perform a shift action.
*/
static void yy_shift(
  yyParser *yypParser,          /* The parser to be shifted */
  int yyNewState,               /* The new state to shift in */
  int yyMajor,                  /* The major token to shift in */
  ParseTOKENTYPE yyMinor        /* The minor token to shift in */
){
  if( yyNewState > YY_MAX_SHIFT ){
    yyNewState += YY_MIN_REDUCE - YY_MIN_SHIFTREDUCE;
  }
  yypParser->yystack.push_back(yyStackEntry(yyNewState, yyMajor, yyMinor));
#ifdef YYTRACKMAXSTACKDEPTH
  if( (int)(yypParser->yystack.size()>yypParser->yyhwm ){
    yypParser->yyhwm++;
    Assert( yypParser->yyhwm == (int)(yypParser->yystack.size() );
  }
#endif
  yyTraceShift(yypParser, yyNewState, "Shift");
}

/* The following table contains information about every rule that
** is used during the reduce.
*/
static const struct {
  YYCODETYPE lhs;       /* Symbol on the left-hand side of the rule */
  signed char nrhs;     /* Negative of the number of RHS symbols in the rule */
} yyRuleInfo[] = {
  {   27,   -1 }, /* (0) query ::= expr */
  {   27,    0 }, /* (1) query ::= */
  {   28,   -3 }, /* (2) expr ::= bool_arg AND bool_arg */
  {   28,   -3 }, /* (3) expr ::= bool_arg NOT bool_arg */
  {   28,   -4 }, /* (4) expr ::= bool_arg AND NOT bool_arg */
  {   28,   -4 }, /* (5) expr ::= bool_arg AND HATE_AFTER_AND bool_arg */
  {   28,   -3 }, /* (6) expr ::= bool_arg OR bool_arg */
  {   28,   -3 }, /* (7) expr ::= bool_arg XOR bool_arg */
  {   28,   -3 }, /* (8) expr ::= bool_arg SYN bool_arg */
  {   30,    0 }, /* (9) bool_arg ::= */
  {   29,   -1 }, /* (10) prob_expr ::= prob */
  {   31,   -1 }, /* (11) prob ::= RANGE */
  {   31,   -2 }, /* (12) prob ::= stop_prob RANGE */
  {   31,   -2 }, /* (13) prob ::= stop_term stop_term */
  {   31,   -2 }, /* (14) prob ::= prob stop_term */
  {   31,   -2 }, /* (15) prob ::= LOVE term */
  {   31,   -3 }, /* (16) prob ::= stop_prob LOVE term */
  {   31,   -2 }, /* (17) prob ::= HATE term */
  {   31,   -3 }, /* (18) prob ::= stop_prob HATE term */
  {   31,   -2 }, /* (19) prob ::= HATE BOOLEAN_FILTER */
  {   31,   -3 }, /* (20) prob ::= stop_prob HATE BOOLEAN_FILTER */
  {   31,   -1 }, /* (21) prob ::= BOOLEAN_FILTER */
  {   31,   -2 }, /* (22) prob ::= stop_prob BOOLEAN_FILTER */
  {   31,   -2 }, /* (23) prob ::= LOVE BOOLEAN_FILTER */
  {   31,   -3 }, /* (24) prob ::= stop_prob LOVE BOOLEAN_FILTER */
  {   33,   -1 }, /* (25) stop_prob ::= stop_term */
  {   34,   -1 }, /* (26) stop_term ::= TERM */
  {   32,   -1 }, /* (27) term ::= TERM */
  {   35,   -1 }, /* (28) compound_term ::= EDIT_TERM */
  {   35,   -1 }, /* (29) compound_term ::= WILD_TERM */
  {   35,   -1 }, /* (30) compound_term ::= PARTIAL_TERM */
  {   35,   -3 }, /* (31) compound_term ::= QUOTE phrase QUOTE */
  {   35,   -1 }, /* (32) compound_term ::= phrased_term */
  {   35,   -1 }, /* (33) compound_term ::= group */
  {   35,   -1 }, /* (34) compound_term ::= near_expr */
  {   35,   -1 }, /* (35) compound_term ::= adj_expr */
  {   35,   -3 }, /* (36) compound_term ::= BRA expr KET */
  {   35,   -2 }, /* (37) compound_term ::= SYNONYM TERM */
  {   35,   -4 }, /* (38) compound_term ::= SYNONYM QUOTE phrase QUOTE */
  {   35,   -1 }, /* (39) compound_term ::= UNBROKEN_WORDS */
  {   36,   -1 }, /* (40) phrase ::= TERM */
  {   36,   -1 }, /* (41) phrase ::= UNBROKEN_WORDS */
  {   36,   -2 }, /* (42) phrase ::= phrase TERM */
  {   36,   -2 }, /* (43) phrase ::= phrase UNBROKEN_WORDS */
  {   37,   -2 }, /* (44) phrased_term ::= TERM PHR_TERM */
  {   37,   -2 }, /* (45) phrased_term ::= phrased_term PHR_TERM */
  {   38,   -2 }, /* (46) group ::= TERM GROUP_TERM */
  {   38,   -2 }, /* (47) group ::= group GROUP_TERM */
  {   38,   -2 }, /* (48) group ::= group EMPTY_GROUP_OK */
  {   39,   -3 }, /* (49) near_expr ::= TERM NEAR TERM */
  {   39,   -3 }, /* (50) near_expr ::= near_expr NEAR TERM */
  {   40,   -3 }, /* (51) adj_expr ::= TERM ADJ TERM */
  {   40,   -3 }, /* (52) adj_expr ::= adj_expr ADJ TERM */
  {   28,   -1 }, /* (53) expr ::= prob_expr */
  {   30,   -1 }, /* (54) bool_arg ::= expr */
  {   29,   -1 }, /* (55) prob_expr ::= term */
  {   33,   -1 }, /* (56) stop_prob ::= prob */
  {   34,   -1 }, /* (57) stop_term ::= compound_term */
  {   32,   -1 }, /* (58) term ::= compound_term */
};

static void yy_accept(yyParser*);  /* Forward Declaration */

/*
** Perform a reduce action and the shift that must immediately
** follow the reduce.
**
** The yyLookahead and yyLookaheadToken parameters provide reduce actions
** access to the lookahead token (if any).  The yyLookahead will be YYNOCODE
** if the lookahead token has already been consumed.  As this procedure is
** only called from one place, optimizing compilers will in-line it, which
** means that the extra parameters have no performance impact.
*/
static void yy_reduce(
  yyParser *yypParser,         /* The parser */
  unsigned int yyruleno,       /* Number of the rule by which to reduce */
  int yyLookahead,             /* Lookahead token, or YYNOCODE if none */
  ParseTOKENTYPE yyLookaheadToken  /* Value of the lookahead token */
){
  int yygoto;                     /* The next state */
  int yyact;                      /* The next action */
  yyStackEntry *yymsp;            /* The top of the parser's stack */
  int yysize;                     /* Amount to pop the stack */
  ParseARG_FETCH;
  (void)yyLookahead;
  (void)yyLookaheadToken;
  yymsp = &yypParser->yystack.back();
  Assert( yyruleno<sizeof(yyRuleInfo)/sizeof(yyRuleInfo[0]) );
#ifdef XAPIAN_DEBUG_LOG
  {
    yysize = yyRuleInfo[yyruleno].nrhs;
    if( yysize ){
      LOGLINE(QUERYPARSER, "Reduce " << yyruleno << " [" <<
                           ParseRuleName(yyruleno) << "], go to state " <<
                           yymsp[yysize].stateno);
    } else {
      LOGLINE(QUERYPARSER, "Reduce " << yyruleno << " [" <<
                           ParseRuleName(yyruleno) << "].");
    }
  }
#endif /* XAPIAN_DEBUG_LOG */
  /*  yygotominor = yyzerominor; */

  /* Check that the stack is large enough to grow by a single entry
  ** if the RHS of the rule is empty.  This ensures that there is room
  ** enough on the stack to push the LHS value without invalidating
  ** pointers into the stack. */
  if( yyRuleInfo[yyruleno].nrhs==0 ){
#if 1
    yypParser->yystack.resize(yypParser->yystack.size() + 1);
    yymsp = &(yypParser->yystack.back()) - 1;
#else
#ifdef YYTRACKMAXSTACKDEPTH
    if( (int)(yypParser->yytos - yypParser->yystack)>yypParser->yyhwm ){
      yypParser->yyhwm++;
      Assert( yypParser->yyhwm == (int)(yypParser->yytos - yypParser->yystack));
    }
#endif
#if YYSTACKDEPTH>0 
    if( yypParser->yytos>=yypParser->yystackEnd ){
      yyStackOverflow(yypParser);
      return;
    }
#else
    if( yypParser->yytos>=&yypParser->yystack[yypParser->yystksz-1] ){
      if( yyGrowStack(yypParser) ){
        yyStackOverflow(yypParser);
        return;
      }
      yymsp = yypParser->yytos;
    }
#endif
#endif
  }

  switch( yyruleno ){
  /* Beginning here are the reduction cases.  A typical example
  ** follows:
  **   case 0:
  **  #line <lineno> <grammarfile>
  **     { ... }           // User supplied code
  **  #line <lineno> <thisfile>
  **     break;
  */
/********** Begin reduce actions **********************************************/
        YYMINORTYPE yylhsminor;
      case 0: /* query ::= expr */
#line 2278 "queryparser/queryparser.lemony"
{
    // Save the parsed query in the State structure so we can return it.
    if (yymsp[0].minor.yy1) {
	state->query = *yymsp[0].minor.yy1;
	delete yymsp[0].minor.yy1;
    } else {
	state->query = Query();
    }
}
#line 2598 "queryparser/queryparser_internal.cc"
        break;
      case 1: /* query ::= */
#line 2288 "queryparser/queryparser.lemony"
{
    // Handle a query string with no terms in.
    state->query = Query();
}
#line 2606 "queryparser/queryparser_internal.cc"
        break;
      case 2: /* expr ::= bool_arg AND bool_arg */
#line 2300 "queryparser/queryparser.lemony"
{
    VET_BOOL_ARGS(yymsp[-2].minor.yy1, yymsp[0].minor.yy1, "AND");
    *yymsp[-2].minor.yy1 &= *yymsp[0].minor.yy1;
    delete yymsp[0].minor.yy1;
}
#line 2615 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,4,&yymsp[-1].minor);
        break;
      case 3: /* expr ::= bool_arg NOT bool_arg */
#line 2306 "queryparser/queryparser.lemony"
{
    if (!yymsp[-2].minor.yy1 && (state->flags & QueryParser::FLAG_PURE_NOT)) {
	// 'NOT foo' -> '(0 * <alldocuments>) NOT foo'
	//
	// We scale the <alldocuments> by 0 so it doesn't count towards the
	// number of matching subqueries since that allows the query optimiser
	// to eliminate it if other subqueries are combined in an AND-like
	// way (e.g. 'bar AND (NOT foo)').
	yymsp[-2].minor.yy1 = new Query(0.0, Query(string(), 1, 0));
    }
    VET_BOOL_ARGS(yymsp[-2].minor.yy1, yymsp[0].minor.yy1, "NOT");
    *yymsp[-2].minor.yy1 &= ~*yymsp[0].minor.yy1;
    delete yymsp[0].minor.yy1;
}
#line 2634 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,5,&yymsp[-1].minor);
        break;
      case 4: /* expr ::= bool_arg AND NOT bool_arg */
#line 2321 "queryparser/queryparser.lemony"
{
    VET_BOOL_ARGS(yymsp[-3].minor.yy1, yymsp[0].minor.yy1, "AND NOT");
    *yymsp[-3].minor.yy1 &= ~*yymsp[0].minor.yy1;
    delete yymsp[0].minor.yy1;
}
#line 2644 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,4,&yymsp[-2].minor);
  yy_destructor(yypParser,5,&yymsp[-1].minor);
        break;
      case 5: /* expr ::= bool_arg AND HATE_AFTER_AND bool_arg */
#line 2327 "queryparser/queryparser.lemony"
{
    VET_BOOL_ARGS(yymsp[-3].minor.yy1, yymsp[0].minor.yy1, "AND");
    *yymsp[-3].minor.yy1 &= ~*yymsp[0].minor.yy1;
    delete yymsp[0].minor.yy1;
}
#line 2655 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,4,&yymsp[-2].minor);
  yy_destructor(yypParser,10,&yymsp[-1].minor);
        break;
      case 6: /* expr ::= bool_arg OR bool_arg */
#line 2333 "queryparser/queryparser.lemony"
{
    VET_BOOL_ARGS(yymsp[-2].minor.yy1, yymsp[0].minor.yy1, "OR");
    *yymsp[-2].minor.yy1 |= *yymsp[0].minor.yy1;
    delete yymsp[0].minor.yy1;
}
#line 2666 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,2,&yymsp[-1].minor);
        break;
      case 7: /* expr ::= bool_arg XOR bool_arg */
#line 2339 "queryparser/queryparser.lemony"
{
    VET_BOOL_ARGS(yymsp[-2].minor.yy1, yymsp[0].minor.yy1, "XOR");
    *yymsp[-2].minor.yy1 ^= *yymsp[0].minor.yy1;
    delete yymsp[0].minor.yy1;
}
#line 2676 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,3,&yymsp[-1].minor);
        break;
      case 8: /* expr ::= bool_arg SYN bool_arg */
#line 2345 "queryparser/queryparser.lemony"
{
    VET_BOOL_ARGS(yymsp[-2].minor.yy1, yymsp[0].minor.yy1, "SYN");
    *yymsp[-2].minor.yy1 = Query(Query::OP_SYNONYM, *yymsp[-2].minor.yy1, *yymsp[0].minor.yy1);
    delete yymsp[0].minor.yy1;
}
#line 2686 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,12,&yymsp[-1].minor);
        break;
      case 9: /* bool_arg ::= */
#line 2358 "queryparser/queryparser.lemony"
{
    // Set the argument to NULL, which enables the bool_arg-using rules in
    // expr above to report uses of AND, OR, etc which don't have two
    // arguments.
    yymsp[1].minor.yy1 = NULL;
}
#line 2697 "queryparser/queryparser_internal.cc"
        break;
      case 10: /* prob_expr ::= prob */
#line 2370 "queryparser/queryparser.lemony"
{
    yylhsminor.yy1 = yymsp[0].minor.yy18->query;
    yymsp[0].minor.yy18->query = NULL;
    // Handle any "+ terms".
    if (yymsp[0].minor.yy18->love) {
	if (yymsp[0].minor.yy18->love->empty()) {
	    // +<nothing>.
	    delete yylhsminor.yy1;
	    yylhsminor.yy1 = yymsp[0].minor.yy18->love;
	} else if (yylhsminor.yy1) {
	    swap(yylhsminor.yy1, yymsp[0].minor.yy18->love);
	    add_to_query(yylhsminor.yy1, Query::OP_AND_MAYBE, yymsp[0].minor.yy18->love);
	} else {
	    yylhsminor.yy1 = yymsp[0].minor.yy18->love;
	}
	yymsp[0].minor.yy18->love = NULL;
    }
    // Handle any boolean filters.
    if (!yymsp[0].minor.yy18->filter.empty()) {
	if (yylhsminor.yy1) {
	    add_to_query(yylhsminor.yy1, Query::OP_FILTER, yymsp[0].minor.yy18->merge_filters());
	} else {
	    // Make the query a boolean one.
	    yylhsminor.yy1 = new Query(Query::OP_SCALE_WEIGHT, yymsp[0].minor.yy18->merge_filters(), 0.0);
	}
    }
    // Handle any "- terms".
    if (yymsp[0].minor.yy18->hate && !yymsp[0].minor.yy18->hate->empty()) {
	if (!yylhsminor.yy1) {
	    // Can't just hate!
	    yy_parse_failed(yypParser);
	    return;
	}
	*yylhsminor.yy1 = Query(Query::OP_AND_NOT, *yylhsminor.yy1, *yymsp[0].minor.yy18->hate);
    }
    delete yymsp[0].minor.yy18;
}
#line 2738 "queryparser/queryparser_internal.cc"
  yymsp[0].minor.yy1 = yylhsminor.yy1;
        break;
      case 11: /* prob ::= RANGE */
#line 2418 "queryparser/queryparser.lemony"
{
    string grouping = yymsp[0].minor.yy0->name;
    const Query & range = yymsp[0].minor.yy0->as_range_query();
    yymsp[0].minor.yy18 = new ProbQuery; /*P-overwrites-R*/
    yymsp[0].minor.yy18->add_filter_range(grouping, range);
}
#line 2749 "queryparser/queryparser_internal.cc"
        break;
      case 12: /* prob ::= stop_prob RANGE */
#line 2425 "queryparser/queryparser.lemony"
{
    string grouping = yymsp[0].minor.yy0->name;
    const Query & range = yymsp[0].minor.yy0->as_range_query();
    yymsp[-1].minor.yy18->append_filter_range(grouping, range);
}
#line 2758 "queryparser/queryparser_internal.cc"
        break;
      case 13: /* prob ::= stop_term stop_term */
#line 2431 "queryparser/queryparser.lemony"
{
    yymsp[-1].minor.yy18 = new ProbQuery(yymsp[-1].minor.yy1); /*P-overwrites-T*/
    if (yymsp[0].minor.yy1) {
	Query::op op = state->default_op();
	if (yymsp[-1].minor.yy18->query && is_positional(op)) {
	    // If default_op is OP_NEAR or OP_PHRASE, set the window size to
	    // 11 for the first pair of terms and it will automatically grow
	    // by one for each subsequent term.
	    Query * subqs[2] = { yymsp[-1].minor.yy18->query, yymsp[0].minor.yy1 };
	    *(yymsp[-1].minor.yy18->query) = Query(op, subqs, subqs + 2, 11);
	    delete yymsp[0].minor.yy1;
	} else {
	    add_to_query(yymsp[-1].minor.yy18->query, op, yymsp[0].minor.yy1);
	}
    }
}
#line 2778 "queryparser/queryparser_internal.cc"
        break;
      case 14: /* prob ::= prob stop_term */
#line 2448 "queryparser/queryparser.lemony"
{
    // If yymsp[0].minor.yy1 is a stopword, there's nothing to do here.
    if (yymsp[0].minor.yy1) add_to_query(yymsp[-1].minor.yy18->query, state->default_op(), yymsp[0].minor.yy1);
}
#line 2786 "queryparser/queryparser_internal.cc"
        break;
      case 15: /* prob ::= LOVE term */
{  yy_destructor(yypParser,8,&yymsp[-1].minor);
#line 2453 "queryparser/queryparser.lemony"
{
    yymsp[-1].minor.yy18 = new ProbQuery;
    if (state->default_op() == Query::OP_AND) {
	yymsp[-1].minor.yy18->query = yymsp[0].minor.yy1;
    } else {
	yymsp[-1].minor.yy18->love = yymsp[0].minor.yy1;
    }
}
#line 2799 "queryparser/queryparser_internal.cc"
}
        break;
      case 16: /* prob ::= stop_prob LOVE term */
#line 2462 "queryparser/queryparser.lemony"
{
    if (state->default_op() == Query::OP_AND) {
	/* The default op is AND, so we just put loved terms into the query
	 * (in this case the only effect of love is to ignore the stopword
	 * list). */
	add_to_query(yymsp[-2].minor.yy18->query, Query::OP_AND, yymsp[0].minor.yy1);
    } else {
	add_to_query(yymsp[-2].minor.yy18->love, Query::OP_AND, yymsp[0].minor.yy1);
    }
}
#line 2814 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,8,&yymsp[-1].minor);
        break;
      case 17: /* prob ::= HATE term */
{  yy_destructor(yypParser,9,&yymsp[-1].minor);
#line 2473 "queryparser/queryparser.lemony"
{
    yymsp[-1].minor.yy18 = new ProbQuery;
    yymsp[-1].minor.yy18->hate = yymsp[0].minor.yy1;
}
#line 2824 "queryparser/queryparser_internal.cc"
}
        break;
      case 18: /* prob ::= stop_prob HATE term */
#line 2478 "queryparser/queryparser.lemony"
{
    add_to_query(yymsp[-2].minor.yy18->hate, Query::OP_OR, yymsp[0].minor.yy1);
}
#line 2832 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,9,&yymsp[-1].minor);
        break;
      case 19: /* prob ::= HATE BOOLEAN_FILTER */
{  yy_destructor(yypParser,9,&yymsp[-1].minor);
#line 2482 "queryparser/queryparser.lemony"
{
    yymsp[-1].minor.yy18 = new ProbQuery;
    yymsp[-1].minor.yy18->hate = new Query(yymsp[0].minor.yy0->get_query());
    delete yymsp[0].minor.yy0;
}
#line 2843 "queryparser/queryparser_internal.cc"
}
        break;
      case 20: /* prob ::= stop_prob HATE BOOLEAN_FILTER */
#line 2488 "queryparser/queryparser.lemony"
{
    add_to_query(yymsp[-2].minor.yy18->hate, Query::OP_OR, yymsp[0].minor.yy0->get_query());
    delete yymsp[0].minor.yy0;
}
#line 2852 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,9,&yymsp[-1].minor);
        break;
      case 21: /* prob ::= BOOLEAN_FILTER */
#line 2493 "queryparser/queryparser.lemony"
{
    yylhsminor.yy18 = new ProbQuery;
    yylhsminor.yy18->add_filter(yymsp[0].minor.yy0->get_grouping(), yymsp[0].minor.yy0->get_query());
    delete yymsp[0].minor.yy0;
}
#line 2862 "queryparser/queryparser_internal.cc"
  yymsp[0].minor.yy18 = yylhsminor.yy18;
        break;
      case 22: /* prob ::= stop_prob BOOLEAN_FILTER */
#line 2499 "queryparser/queryparser.lemony"
{
    yymsp[-1].minor.yy18->append_filter(yymsp[0].minor.yy0->get_grouping(), yymsp[0].minor.yy0->get_query());
    delete yymsp[0].minor.yy0;
}
#line 2871 "queryparser/queryparser_internal.cc"
        break;
      case 23: /* prob ::= LOVE BOOLEAN_FILTER */
{  yy_destructor(yypParser,8,&yymsp[-1].minor);
#line 2504 "queryparser/queryparser.lemony"
{
    // LOVE BOOLEAN_FILTER(yymsp[0].minor.yy0) is just the same as BOOLEAN_FILTER
    yymsp[-1].minor.yy18 = new ProbQuery;
    yymsp[-1].minor.yy18->filter[yymsp[0].minor.yy0->get_grouping()] = yymsp[0].minor.yy0->get_query();
    delete yymsp[0].minor.yy0;
}
#line 2882 "queryparser/queryparser_internal.cc"
}
        break;
      case 24: /* prob ::= stop_prob LOVE BOOLEAN_FILTER */
#line 2511 "queryparser/queryparser.lemony"
{
    // LOVE BOOLEAN_FILTER(yymsp[0].minor.yy0) is just the same as BOOLEAN_FILTER
    // We OR filters with the same prefix...
    Query & q = yymsp[-2].minor.yy18->filter[yymsp[0].minor.yy0->get_grouping()];
    q |= yymsp[0].minor.yy0->get_query();
    delete yymsp[0].minor.yy0;
}
#line 2894 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,8,&yymsp[-1].minor);
        break;
      case 25: /* stop_prob ::= stop_term */
#line 2526 "queryparser/queryparser.lemony"
{
    yymsp[0].minor.yy18 = new ProbQuery(yymsp[0].minor.yy1); /*P-overwrites-T*/
}
#line 2902 "queryparser/queryparser_internal.cc"
        break;
      case 26: /* stop_term ::= TERM */
#line 2539 "queryparser/queryparser.lemony"
{
    if (state->is_stopword(yymsp[0].minor.yy0)) {
	yylhsminor.yy1 = NULL;
	state->add_to_stoplist(yymsp[0].minor.yy0);
    } else {
	yylhsminor.yy1 = new Query(yymsp[0].minor.yy0->get_query_with_auto_synonyms());
    }
    delete yymsp[0].minor.yy0;
}
#line 2915 "queryparser/queryparser_internal.cc"
  yymsp[0].minor.yy1 = yylhsminor.yy1;
        break;
      case 27: /* term ::= TERM */
#line 2556 "queryparser/queryparser.lemony"
{
    yylhsminor.yy1 = new Query(yymsp[0].minor.yy0->get_query_with_auto_synonyms());
    delete yymsp[0].minor.yy0;
}
#line 2924 "queryparser/queryparser_internal.cc"
  yymsp[0].minor.yy1 = yylhsminor.yy1;
        break;
      case 28: /* compound_term ::= EDIT_TERM */
#line 2571 "queryparser/queryparser.lemony"
{ yymsp[0].minor.yy1 = yymsp[0].minor.yy0->as_fuzzy_query(state); /*T-overwrites-U*/ }
#line 2930 "queryparser/queryparser_internal.cc"
        break;
      case 29: /* compound_term ::= WILD_TERM */
#line 2574 "queryparser/queryparser.lemony"
{ yymsp[0].minor.yy1 = yymsp[0].minor.yy0->as_wildcarded_query(state); /*T-overwrites-U*/ }
#line 2935 "queryparser/queryparser_internal.cc"
        break;
      case 30: /* compound_term ::= PARTIAL_TERM */
#line 2577 "queryparser/queryparser.lemony"
{ yymsp[0].minor.yy1 = yymsp[0].minor.yy0->as_partial_query(state); /*T-overwrites-U*/ }
#line 2940 "queryparser/queryparser_internal.cc"
        break;
      case 31: /* compound_term ::= QUOTE phrase QUOTE */
{  yy_destructor(yypParser,21,&yymsp[-2].minor);
#line 2580 "queryparser/queryparser.lemony"
{ yymsp[-2].minor.yy1 = yymsp[-1].minor.yy36->as_phrase_query(); }
#line 2946 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,21,&yymsp[0].minor);
}
        break;
      case 32: /* compound_term ::= phrased_term */
#line 2583 "queryparser/queryparser.lemony"
{ yymsp[0].minor.yy1 = yymsp[0].minor.yy36->as_phrase_query(); /*T-overwrites-P*/ }
#line 2953 "queryparser/queryparser_internal.cc"
        break;
      case 33: /* compound_term ::= group */
#line 2586 "queryparser/queryparser.lemony"
{ yymsp[0].minor.yy1 = yymsp[0].minor.yy32->as_group(state); /*T-overwrites-P*/ }
#line 2958 "queryparser/queryparser_internal.cc"
        break;
      case 34: /* compound_term ::= near_expr */
#line 2589 "queryparser/queryparser.lemony"
{ yymsp[0].minor.yy1 = yymsp[0].minor.yy36->as_near_query(); /*T-overwrites-P*/ }
#line 2963 "queryparser/queryparser_internal.cc"
        break;
      case 35: /* compound_term ::= adj_expr */
#line 2592 "queryparser/queryparser.lemony"
{ yymsp[0].minor.yy1 = yymsp[0].minor.yy36->as_adj_query(); /*T-overwrites-P*/ }
#line 2968 "queryparser/queryparser_internal.cc"
        break;
      case 36: /* compound_term ::= BRA expr KET */
{  yy_destructor(yypParser,22,&yymsp[-2].minor);
#line 2595 "queryparser/queryparser.lemony"
{ yymsp[-2].minor.yy1 = yymsp[-1].minor.yy1; }
#line 2974 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,23,&yymsp[0].minor);
}
        break;
      case 37: /* compound_term ::= SYNONYM TERM */
{  yy_destructor(yypParser,11,&yymsp[-1].minor);
#line 2597 "queryparser/queryparser.lemony"
{
    yymsp[-1].minor.yy1 = new Query(yymsp[0].minor.yy0->get_query_with_synonyms());
    delete yymsp[0].minor.yy0;
}
#line 2985 "queryparser/queryparser_internal.cc"
}
        break;
      case 38: /* compound_term ::= SYNONYM QUOTE phrase QUOTE */
{  yy_destructor(yypParser,11,&yymsp[-3].minor);
#line 2603 "queryparser/queryparser.lemony"
{ yymsp[-3].minor.yy1 = yymsp[-1].minor.yy36->as_synonym_phrase_query(state); }
#line 2992 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,21,&yymsp[-2].minor);
  yy_destructor(yypParser,21,&yymsp[0].minor);
}
        break;
      case 39: /* compound_term ::= UNBROKEN_WORDS */
#line 2605 "queryparser/queryparser.lemony"
{
    { yymsp[0].minor.yy1 = yymsp[0].minor.yy0->as_unbroken_query(); /*T-overwrites-U*/ }
}
#line 3002 "queryparser/queryparser_internal.cc"
        break;
      case 40: /* phrase ::= TERM */
#line 2615 "queryparser/queryparser.lemony"
{
    yylhsminor.yy36 = Terms::create(state);
    yylhsminor.yy36->add_positional_term(yymsp[0].minor.yy0);
}
#line 3010 "queryparser/queryparser_internal.cc"
  yymsp[0].minor.yy36 = yylhsminor.yy36;
        break;
      case 41: /* phrase ::= UNBROKEN_WORDS */
#line 2620 "queryparser/queryparser.lemony"
{
    yylhsminor.yy36 = Terms::create(state);
    yymsp[0].minor.yy0->as_positional_unbroken(yylhsminor.yy36);
}
#line 3019 "queryparser/queryparser_internal.cc"
  yymsp[0].minor.yy36 = yylhsminor.yy36;
        break;
      case 42: /* phrase ::= phrase TERM */
      case 45: /* phrased_term ::= phrased_term PHR_TERM */ yytestcase(yyruleno==45);
#line 2625 "queryparser/queryparser.lemony"
{
    yymsp[-1].minor.yy36->add_positional_term(yymsp[0].minor.yy0);
}
#line 3028 "queryparser/queryparser_internal.cc"
        break;
      case 43: /* phrase ::= phrase UNBROKEN_WORDS */
#line 2629 "queryparser/queryparser.lemony"
{
    yymsp[0].minor.yy0->as_positional_unbroken(yymsp[-1].minor.yy36);
}
#line 3035 "queryparser/queryparser_internal.cc"
        break;
      case 44: /* phrased_term ::= TERM PHR_TERM */
#line 2640 "queryparser/queryparser.lemony"
{
    yylhsminor.yy36 = Terms::create(state);
    yylhsminor.yy36->add_positional_term(yymsp[-1].minor.yy0);
    yylhsminor.yy36->add_positional_term(yymsp[0].minor.yy0);
}
#line 3044 "queryparser/queryparser_internal.cc"
  yymsp[-1].minor.yy36 = yylhsminor.yy36;
        break;
      case 46: /* group ::= TERM GROUP_TERM */
#line 2656 "queryparser/queryparser.lemony"
{
    yymsp[-1].minor.yy32 = TermGroup::create(yymsp[-1].minor.yy0, yymsp[0].minor.yy0); /*P-overwrites-T*/
}
#line 3052 "queryparser/queryparser_internal.cc"
        break;
      case 47: /* group ::= group GROUP_TERM */
#line 2660 "queryparser/queryparser.lemony"
{
    yymsp[-1].minor.yy32->add_term(yymsp[0].minor.yy0);
}
#line 3059 "queryparser/queryparser_internal.cc"
        break;
      case 48: /* group ::= group EMPTY_GROUP_OK */
#line 2664 "queryparser/queryparser.lemony"
{
    yymsp[-1].minor.yy32->set_empty_ok();
}
#line 3066 "queryparser/queryparser_internal.cc"
  yy_destructor(yypParser,25,&yymsp[0].minor);
        break;
      case 49: /* near_expr ::= TERM NEAR TERM */
      case 51: /* adj_expr ::= TERM ADJ TERM */ yytestcase(yyruleno==51);
#line 2674 "queryparser/queryparser.lemony"
{
    yylhsminor.yy36 = Terms::create(state);
    yylhsminor.yy36->add_positional_term(yymsp[-2].minor.yy0);
    yylhsminor.yy36->add_positional_term(yymsp[0].minor.yy0);
    if (yymsp[-1].minor.yy0) {
	yylhsminor.yy36->adjust_window(yymsp[-1].minor.yy0->get_termpos());
	delete yymsp[-1].minor.yy0;
    }
}
#line 3081 "queryparser/queryparser_internal.cc"
  yymsp[-2].minor.yy36 = yylhsminor.yy36;
        break;
      case 50: /* near_expr ::= near_expr NEAR TERM */
      case 52: /* adj_expr ::= adj_expr ADJ TERM */ yytestcase(yyruleno==52);
#line 2684 "queryparser/queryparser.lemony"
{
    yymsp[-2].minor.yy36->add_positional_term(yymsp[0].minor.yy0);
    if (yymsp[-1].minor.yy0) {
	yymsp[-2].minor.yy36->adjust_window(yymsp[-1].minor.yy0->get_termpos());
	delete yymsp[-1].minor.yy0;
    }
}
#line 3094 "queryparser/queryparser_internal.cc"
        break;
      default:
      /* (53) expr ::= prob_expr (OPTIMIZED OUT) */ Assert(yyruleno!=53);
      /* (54) bool_arg ::= expr */ yytestcase(yyruleno==54);
      /* (55) prob_expr ::= term (OPTIMIZED OUT) */ Assert(yyruleno!=55);
      /* (56) stop_prob ::= prob */ yytestcase(yyruleno==56);
      /* (57) stop_term ::= compound_term */ yytestcase(yyruleno==57);
      /* (58) term ::= compound_term */ yytestcase(yyruleno==58);
        break;
/********** End reduce actions ************************************************/
  }
  Assert( yyruleno<sizeof(yyRuleInfo)/sizeof(yyRuleInfo[0]) );
  yygoto = yyRuleInfo[yyruleno].lhs;
  yysize = yyRuleInfo[yyruleno].nrhs;
  yyact = yy_find_reduce_action(yymsp[yysize].stateno,static_cast<YYCODETYPE>(yygoto));

  /* There are no SHIFTREDUCE actions on nonterminals because the table
  ** generator has simplified them to pure REDUCE actions. */
  Assert( !(yyact>YY_MAX_SHIFT && yyact<=YY_MAX_SHIFTREDUCE) );

  /* It is not possible for a REDUCE to be followed by an error */
  Assert( yyact!=YY_ERROR_ACTION );

  yymsp += yysize+1;
  if (yysize) {
    yypParser->yystack.resize(UNSIGNED_OVERFLOW_OK(yypParser->yystack.size() +
						   (yysize+1)));
  }
  yymsp->stateno = static_cast<YYACTIONTYPE>(yyact);
  yymsp->major = static_cast<YYCODETYPE>(yygoto);
  yyTraceShift(yypParser, yyact, "... then shift");
}

/*
** The following code executes when the parse fails
*/
#ifndef YYNOERRORRECOVERY
static void yy_parse_failed(
  yyParser *yypParser           /* The parser */
){
  ParseARG_FETCH;
  LOGLINE(QUERYPARSER, "Fail!");
  while( yypParser->yystack.size() > 1 ) yy_pop_parser_stack(yypParser);
  /* Here code is inserted which will be executed whenever the
  ** parser fails */
/************ Begin %parse_failure code ***************************************/
#line 2222 "queryparser/queryparser.lemony"

    // If we've not already set an error message, set a default one.
    if (!state->error) state->error = "parse error";
#line 3145 "queryparser/queryparser_internal.cc"
/************ End %parse_failure code *****************************************/
  ParseARG_STORE; /* Suppress warning about unused %extra_argument variable */
}
#endif /* YYNOERRORRECOVERY */

/*
** The following code executes when a syntax error first occurs.
*/
static void yy_syntax_error(
  yyParser *yypParser,           /* The parser */
  int yymajor,                   /* The major type of the error token */
  ParseTOKENTYPE yyminor         /* The minor type of the error token */
){
  ParseARG_FETCH;
  (void)yymajor;
  (void)yyminor;
#define TOKEN yyminor
/************ Begin %syntax_error code ****************************************/
#line 2227 "queryparser/queryparser.lemony"

    yy_parse_failed(yypParser);
#line 3167 "queryparser/queryparser_internal.cc"
/************ End %syntax_error code ******************************************/
  ParseARG_STORE; /* Suppress warning about unused %extra_argument variable */
}

/*
** The following is executed when the parser accepts
*/
static void yy_accept(
  yyParser *yypParser           /* The parser */
){
  ParseARG_FETCH;
  LOGLINE(QUERYPARSER, "Accept!");
#ifndef YYNOERRORRECOVERY
  yypParser->yyerrcnt = -1;
#endif
  AssertEq( yypParser->yystack.size(), 1 );
  /* Here code is inserted which will be executed whenever the
  ** parser accepts */
/*********** Begin %parse_accept code *****************************************/
/*********** End %parse_accept code *******************************************/
  ParseARG_STORE; /* Suppress warning about unused %extra_argument variable */
}

/* The main parser program.
** The first argument is a pointer to a structure obtained from
** "ParseAlloc" which describes the current state of the parser.
** The second argument is the major token number.  The third is
** the minor token.  The fourth optional argument is whatever the
** user wants (and specified in the grammar) and is available for
** use by the action routines.
**
** Inputs:
** <ul>
** <li> A pointer to the parser (an opaque structure.)
** <li> The major token number.
** <li> The minor token number.
** <li> An option argument of a grammar-specified type.
** </ul>
**
** Outputs:
** None.
*/
static
void Parse(
  yyParser *yypParser,         /* The parser */
  int yymajor,                 /* The major token code number */
  ParseTOKENTYPE yyminor       /* The value for the token */
  ParseARG_PDECL               /* Optional %extra_argument parameter */
){
  YYMINORTYPE yyminorunion;
  unsigned int yyact;   /* The parser action. */
#if !defined(YYERRORSYMBOL) && !defined(YYNOERRORRECOVERY)
  int yyendofinput;     /* True if we are at the end of input */
#endif
#ifdef YYERRORSYMBOL
  int yyerrorhit = 0;   /* True if yymajor has invoked an error */
#endif

#if !defined(YYERRORSYMBOL) && !defined(YYNOERRORRECOVERY)
  yyendofinput = (yymajor==0);
#endif
  ParseARG_STORE;

#ifdef XAPIAN_DEBUG_LOG
  {
    int stateno = yypParser->yystack.back().stateno;
    if( stateno < YY_MIN_REDUCE ){
      LOGLINE(QUERYPARSER, "Input '" << ParseTokenName(yymajor) <<
                           "'," << (yyminor ? yyminor->name : "<<null>>") <<
                           "in state " << stateno);
    }else{
      LOGLINE(QUERYPARSER, "Input '" << ParseTokenName(yymajor) <<
                           "'," << (yyminor ? yyminor->name : "<<null>>") <<
                           "with pending reduce " << stateno-YY_MIN_REDUCE);
    }
  }
#endif

  do{
    yyact = yy_find_shift_action(yypParser,static_cast<YYCODETYPE>(yymajor));
    if( yyact >= YY_MIN_REDUCE ){
      yy_reduce(yypParser,yyact-YY_MIN_REDUCE,yymajor,yyminor);
    }else if( yyact <= YY_MAX_SHIFTREDUCE ){
      yy_shift(yypParser,yyact,yymajor,yyminor);
#ifndef YYNOERRORRECOVERY
      yypParser->yyerrcnt--;
#endif
      yymajor = YYNOCODE;
    }else if( yyact==YY_ACCEPT_ACTION ){
      yypParser->yystack.pop_back();
      yy_accept(yypParser);
      return;
    }else{
      Assert( yyact == YY_ERROR_ACTION );
      yyminorunion.yy0 = yyminor;
#ifdef YYERRORSYMBOL
      int yymx;
#endif
      LOGLINE(QUERYPARSER, "Syntax Error!");
#ifdef YYERRORSYMBOL
      /* A syntax error has occurred.
      ** The response to an error depends upon whether or not the
      ** grammar defines an error token "ERROR".  
      **
      ** This is what we do if the grammar does define ERROR:
      **
      **  * Call the %syntax_error function.
      **
      **  * Begin popping the stack until we enter a state where
      **    it is legal to shift the error symbol, then shift
      **    the error symbol.
      **
      **  * Set the error count to three.
      **
      **  * Begin accepting and shifting new tokens.  No new error
      **    processing will occur until three tokens have been
      **    shifted successfully.
      **
      */
      if( yypParser->yyerrcnt<0 ){
        yy_syntax_error(yypParser,yymajor,yyminor);
      }
      yymx = yypParser->yystack.back().major;
      if( yymx==YYERRORSYMBOL || yyerrorhit ){
        LOGLINE(QUERYPARSER, "Discard input token " << ParseTokenName(yymajor));
        yy_destructor(yypParser, static_cast<YYCODETYPE>(yymajor), &yyminorunion);
        yymajor = YYNOCODE;
      }else{
        while( !yypParser->yystack.empty()
            && yymx != YYERRORSYMBOL
            && (yyact = yy_find_reduce_action(
                        yypParser->yystack.back().stateno,
                        YYERRORSYMBOL)) >= YY_MIN_REDUCE
        ){
          yy_pop_parser_stack(yypParser);
        }
        if( yypParser->yystack.empty() || yymajor==0 ){
          yy_destructor(yypParser,static_cast<YYCODETYPE>(yymajor),&yyminorunion);
          yy_parse_failed(yypParser);
#ifndef YYNOERRORRECOVERY
          yypParser->yyerrcnt = -1;
#endif
          yymajor = YYNOCODE;
        }else if( yymx!=YYERRORSYMBOL ){
          yy_shift(yypParser,yyact,YYERRORSYMBOL,yyminor);
        }
      }
      yypParser->yyerrcnt = 3;
      yyerrorhit = 1;
#elif defined(YYNOERRORRECOVERY)
      /* If the YYNOERRORRECOVERY macro is defined, then do not attempt to
      ** do any kind of error recovery.  Instead, simply invoke the syntax
      ** error routine and continue going as if nothing had happened.
      **
      ** Applications can set this macro (for example inside %include) if
      ** they intend to abandon the parse upon the first syntax error seen.
      */
      yy_syntax_error(yypParser,yymajor, yyminor);
      yy_destructor(yypParser,static_cast<YYCODETYPE>(yymajor),&yyminorunion);
      yymajor = YYNOCODE;
      
#else  /* YYERRORSYMBOL is not defined */
      /* This is what we do if the grammar does not define ERROR:
      **
      **  * Report an error message, and throw away the input token.
      **
      **  * If the input token is $, then fail the parse.
      **
      ** As before, subsequent error messages are suppressed until
      ** three input tokens have been successfully shifted.
      */
      if( yypParser->yyerrcnt<=0 ){
        yy_syntax_error(yypParser,yymajor, yyminor);
      }
      yypParser->yyerrcnt = 3;
      yy_destructor(yypParser,static_cast<YYCODETYPE>(yymajor),&yyminorunion);
      if( yyendofinput ){
        yy_parse_failed(yypParser);
#ifndef YYNOERRORRECOVERY
        yypParser->yyerrcnt = -1;
#endif
      }
      yymajor = YYNOCODE;
#endif
    }
  }while( yymajor!=YYNOCODE && yypParser->yystack.size() > 1 );
#ifdef XAPIAN_DEBUG_LOG
  {
    int i;
    LOGLINE(QUERYPARSER, "Return. Stack=");
    for(i=1; i<=(int)yypParser->yystack.size(); i++)
      LOGLINE(QUERYPARSER, yyTokenName[yypParser->yystack[i].major]);
  }
#endif
  return;
}

// Select C++ syntax highlighting in vim editor: vim: syntax=cpp
#line 1089 "queryparser/queryparser.lemony"


Query
QueryParser::Internal::parse_query(string_view qs, unsigned flags,
				   string_view default_prefix)
{
#ifndef USE_ICU
    // Overall it seems best to check for this up front - otherwise we create
    // the unhelpful situation where a failure to enable ICU in the build could
    // be missed because queries in scripts which don't need word splitting
    // still work fine.
    if (flags & FLAG_WORD_BREAKS) {
	throw Xapian::FeatureUnavailableError("FLAG_WORD_BREAKS requires "
					      "building Xapian to use ICU");
    }
#endif
    bool try_word_break =
	(flags & (FLAG_NGRAMS|FLAG_WORD_BREAKS)) || is_ngram_enabled();

    // Set ranges if we may have to handle ranges in the query.
    bool ranges = !rangeprocs.empty() && (qs.find("..") != string::npos);

    termpos term_pos = 1;
    Utf8Iterator it(qs), end;

    State state(this, flags);

    // To successfully apply more than one spelling correction to a query
    // string, we must keep track of the offset due to previous corrections.
    int correction_offset = 0;
    corrected_query.resize(0);

    // Stack of prefixes, used for phrases and subexpressions.
    list<const FieldInfo *> prefix_stack;

    // If default_prefix is specified, use it.  Otherwise, use any list
    // that has been set for the empty prefix.
    const FieldInfo def_pfx = FieldInfo{NON_BOOLEAN}.append(default_prefix);
    {
	const FieldInfo * default_field_info = &def_pfx;
	if (default_prefix.empty()) {
	    auto f = field_map.find(string_view{});
	    if (f != field_map.end()) default_field_info = &(f->second);
	}

	// We always have the current prefix on the top of the stack.
	prefix_stack.push_back(default_field_info);
    }

    yyParser parser;

    unsigned newprev = ' ';
main_lex_loop:
    enum {
	DEFAULT, IN_QUOTES, IN_PREFIXED_QUOTES, IN_PHRASED_TERM, IN_GROUP,
	IN_GROUP2, EXPLICIT_SYNONYM
    } mode = DEFAULT;
    while (it != end && !state.error) {
	bool last_was_operator = false;
	bool last_was_operator_needing_term = false;
	if (mode == EXPLICIT_SYNONYM) mode = DEFAULT;
	if (false) {
just_had_operator:
	    if (it == end) break;
	    mode = DEFAULT;
	    last_was_operator_needing_term = false;
	    last_was_operator = true;
	}
	if (false) {
just_had_operator_needing_term:
	    last_was_operator_needing_term = true;
	    last_was_operator = true;
	}
	if (mode == IN_PHRASED_TERM) mode = DEFAULT;
	if (is_whitespace(*it)) {
	    newprev = ' ';
	    ++it;
	    it = find_if(it, end, is_not_whitespace);
	    if (it == end) break;
	}

	if (ranges &&
	    (mode == DEFAULT || mode == IN_GROUP || mode == IN_GROUP2)) {
	    // Scan forward to see if this could be the "start of range"
	    // token.  Sadly this has O(n²) tendencies, though at least
	    // "n" is the number of words in a query which is likely to
	    // remain fairly small.  FIXME: can we tokenise more elegantly?
	    Utf8Iterator it_initial = it;
	    Utf8Iterator p = it;
	    unsigned ch = 0;
	    while (p != end) {
		if (ch == '.' && *p == '.') {
		    string a;
		    while (it != p) {
			Unicode::append_utf8(a, *it++);
		    }
		    // Trim off the trailing ".".
		    a.resize(a.size() - 1);
		    ++p;
		    // Either end of the range can be empty (for an open-ended
		    // range) but both can't be empty.
		    if (!a.empty() || (p != end && *p > ' ' && *p != ')')) {
			string b;
			// Allow any character except whitespace and ')' in the
			// upper bound.
			while (p != end && *p > ' ' && *p != ')') {
			    Unicode::append_utf8(b, *p++);
			}
			Term * range = state.range(a, b);
			if (!range) {
			    state.error = "Unknown range operation";
			    if (a.find(':', 1) == string::npos) {
				goto done;
			    }
			    // Might be a boolean filter with ".." in.  Leave
			    // state.error in case it isn't.
			    it = it_initial;
			    break;
			}
			Parse(&parser, RANGE, range, &state);
		    }
		    it = p;
		    goto main_lex_loop;
		}
		ch = *p;
		// Allow any character except whitespace and '(' in the lower
		// bound.
		if (ch <= ' ' || ch == '(') break;
		++p;
	    }
	}

	if (!is_wordchar(*it)) {
	    unsigned prev = newprev;
	    Utf8Iterator p = it;
	    unsigned ch = *it++;
	    newprev = ch;
	    // Drop out of IN_GROUP mode.
	    if (mode == IN_GROUP || mode == IN_GROUP2)
		mode = DEFAULT;
	    switch (ch) {
	      case '"':
	      case 0x201c: // Left curly double quote.
	      case 0x201d: // Right curly double quote.
		// Quoted phrase.
		if (mode == DEFAULT) {
		    // Skip whitespace.
		    it = find_if(it, end, is_not_whitespace);
		    if (it == end) {
			// Ignore an unmatched " at the end of the query to
			// avoid generating an empty pair of QUOTEs which will
			// cause a parse error.
			goto done;
		    }
		    if (is_double_quote(*it)) {
			// Ignore empty "" (but only if we're not already
			// IN_QUOTES as we don't merge two adjacent quoted
			// phrases!)
			newprev = *it++;
			break;
		    }
		}
		if (flags & QueryParser::FLAG_PHRASE) {
		    if (ch == '"' && it != end && *it == '"') {
			++it;
			// Handle "" inside a quoted phrase as an escaped " for
			// consistency with quoted boolean terms.
			break;
		    }
		    Parse(&parser, QUOTE, NULL, &state);
		    if (mode == DEFAULT) {
			mode = IN_QUOTES;
		    } else {
			// Remove the prefix we pushed for this phrase.
			if (mode == IN_PREFIXED_QUOTES)
			    prefix_stack.pop_back();
			mode = DEFAULT;
		    }
		}
		break;

	      case '+': case '-': // Loved or hated term/phrase/subexpression.
		// Ignore + or - at the end of the query string.
		if (it == end) goto done;
		if (prev > ' ' && prev != '(') {
		    // Or if not after whitespace or an open bracket.
		    break;
		}
		if (is_whitespace(*it) || *it == '+' || *it == '-') {
		    // Ignore + or - followed by a space, or further + or -.
		    // Postfix + (such as in C++ and H+) is handled as part of
		    // the term lexing code in parse_term().
		    newprev = *it++;
		    break;
		}
		if (mode == DEFAULT && (flags & FLAG_LOVEHATE)) {
		    int token;
		    if (ch == '+') {
			token = LOVE;
		    } else if (last_was_operator) {
			token = HATE_AFTER_AND;
		    } else {
			token = HATE;
		    }
		    Parse(&parser, token, NULL, &state);
		    goto just_had_operator_needing_term;
		}
		// Need to prevent the term after a LOVE or HATE starting a
		// term group...
		break;

	      case '(': // Bracketed subexpression.
		// Skip whitespace.
		it = find_if(it, end, is_not_whitespace);
		// Ignore ( at the end of the query string.
		if (it == end) goto done;
		if (prev > ' ' && strchr("()+-", prev) == NULL) {
		    // Or if not after whitespace or a bracket or '+' or '-'.
		    break;
		}
		if (*it == ')') {
		    // Ignore empty ().
		    newprev = *it++;
		    break;
		}
		if (mode == DEFAULT && (flags & FLAG_BOOLEAN)) {
		    prefix_stack.push_back(prefix_stack.back());
		    Parse(&parser, BRA, NULL, &state);
		}
		break;

	      case ')': // End of bracketed subexpression.
		if (mode == DEFAULT && (flags & FLAG_BOOLEAN)) {
		    // Remove the prefix we pushed for the corresponding BRA.
		    // If brackets are unmatched, it's a syntax error, but
		    // that's no excuse to SEGV!
		    if (prefix_stack.size() > 1) prefix_stack.pop_back();
		    Parse(&parser, KET, NULL, &state);
		}
		break;

	      case '~': // Synonym expansion.
		// Ignore at the end of the query string.
		if (it == end) goto done;
		if (mode == DEFAULT && (flags & FLAG_SYNONYM)) {
		    if (prev > ' ' && strchr("+-(", prev) == NULL) {
			// Or if not after whitespace, +, -, or an open bracket.
			break;
		    }
		    if (!is_wordchar(*it) && !is_double_quote(*it)) {
			// Ignore if not followed by a word character.
			break;
		    }
		    Parse(&parser, SYNONYM, NULL, &state);
		    mode = EXPLICIT_SYNONYM;
		    if (!is_double_quote(*it))
			goto just_had_operator_needing_term;

		    // Support ~"foo bar" syntax to explicitly expand
		    // a multi-word synonym.

		    // Skip whitespace.
		    ++it;
		    it = find_if(it, end, is_not_whitespace);
		    if (it == end) {
			// Ignore an unmatched " at the end of the query to
			// avoid generating an empty pair of QUOTEs which will
			// cause a parse error.
			goto done;
		    }
		    if (is_double_quote(*it)) {
			// Ignore empty ~"".
			newprev = *it++;
			break;
		    }
		    Parse(&parser, QUOTE, NULL, &state);
		    mode = IN_QUOTES;
		}
		break;
	      case '*':
		if (flags & FLAG_WILDCARD_MULTI) {
		    it = p;
		    goto leading_wildcard;
		}
		break;
	      case '?':
		if (flags & FLAG_WILDCARD_SINGLE) {
		    it = p;
		    goto leading_wildcard;
		}
		break;
	    }
	    // Skip any other characters.
	    continue;
	}

	Assert(is_wordchar(*it));

leading_wildcard:
	size_t term_start_index = it.raw() - qs.data();

	newprev = 'A'; // Any letter will do...

	// A term, a prefix, or a boolean operator.
	const FieldInfo * field_info = NULL;
	if ((mode == DEFAULT || mode == IN_GROUP || mode == IN_GROUP2 || mode == EXPLICIT_SYNONYM) &&
	    !field_map.empty()) {
	    // Check for a fieldname prefix (e.g. title:historical).
	    Utf8Iterator p = find_if(it, end, is_not_wordchar);
	    if (p != end && *p == ':' && ++p != end && *p > ' ' && *p != ')') {
		string field;
		p = it;
		while (*p != ':')
		    Unicode::append_utf8(field, *p++);
		auto f = field_map.find(field);
		if (f != field_map.end()) {
		    // Special handling for prefixed fields, depending on the
		    // type of the prefix.
		    unsigned ch = *++p;
		    field_info = &(f->second);

		    if (field_info->type != NON_BOOLEAN) {
			// Drop out of IN_GROUP if we're in it.
			if (mode == IN_GROUP || mode == IN_GROUP2)
			    mode = DEFAULT;
			it = p;
			string name;
			if (it != end && is_double_quote(*it)) {
			    // Quoted boolean term (can contain any character).
			    bool fancy = (*it != '"');
			    ++it;
			    while (it != end) {
				if (*it == '"') {
				    // Interpret "" as an escaped ".
				    if (++it == end || *it != '"')
					break;
				} else if (fancy && is_double_quote(*it)) {
				    // If the opening quote was ASCII, then the
				    // closing one must be too - otherwise
				    // the user can't protect non-ASCII double
				    // quote characters by quoting or escaping.
				    ++it;
				    break;
				}
				Unicode::append_utf8(name, *it++);
			    }
			} else {
			    // Can't boolean filter prefix a subexpression, so
			    // just use anything following the prefix until the
			    // next space or ')' as part of the boolean filter
			    // term.
			    while (it != end && *it > ' ' && *it != ')')
				Unicode::append_utf8(name, *it++);
			}
			// Build the unstemmed form in field.
			field += ':';
			field += name;
			// Clear any pending range error.
			state.error = NULL;
			Term * token = new Term(&state, name, field_info, field);
			Parse(&parser, BOOLEAN_FILTER, token, &state);
			continue;
		    }

		    if ((flags & FLAG_PHRASE) && is_double_quote(ch)) {
			// Prefixed phrase, e.g.: subject:"space flight"
			mode = IN_PREFIXED_QUOTES;
			Parse(&parser, QUOTE, NULL, &state);
			it = p;
			newprev = ch;
			++it;
			prefix_stack.push_back(field_info);
			continue;
		    }

		    if (ch == '(' && (flags & FLAG_BOOLEAN)) {
			// Prefixed subexpression, e.g.: title:(fast NEAR food)
			mode = DEFAULT;
			Parse(&parser, BRA, NULL, &state);
			it = p;
			newprev = ch;
			++it;
			prefix_stack.push_back(field_info);
			continue;
		    }

		    if (ch != ':') {
			// Allow 'path:/usr/local' but not 'foo::bar::baz'.
			while (is_phrase_generator(ch)) {
			    if (++p == end)
				goto not_prefix;
			    ch = *p;
			}
		    }

		    if (is_wordchar(ch)) {
			// Prefixed term.
			it = p;
		    } else {
not_prefix:
			// It looks like a prefix but isn't, so parse it as
			// text instead.
			field_info = NULL;
		    }
		}
	    }
	}

phrased_term:
	bool was_acronym;
	bool needs_word_break = false;
	size_t first_wildcard = string::npos;
	size_t term_char_count;
	unsigned edit_distance = NO_EDIT_DISTANCE;
	string term = parse_term(it, end, try_word_break, flags,
				 needs_word_break, was_acronym, first_wildcard,
				 term_char_count, edit_distance);

	if (first_wildcard == string::npos &&
	    edit_distance == NO_EDIT_DISTANCE &&
	    (mode == DEFAULT || mode == IN_GROUP || mode == IN_GROUP2) &&
	    (flags & FLAG_BOOLEAN) &&
	    // Don't want to interpret A.N.D. as an AND operator.
	    !was_acronym &&
	    !field_info &&
	    term.size() >= 2 && term.size() <= 4 && U_isalpha(term[0])) {
	    // Boolean operators.
	    string op = term;
	    if (flags & FLAG_BOOLEAN_ANY_CASE) {
		for (string::iterator i = op.begin(); i != op.end(); ++i) {
		    *i = C_toupper(*i);
		}
	    }
	    if (op.size() == 3) {
		if (op == "AND") {
		    Parse(&parser, AND, NULL, &state);
		    goto just_had_operator;
		}
		if (op == "NOT") {
		    Parse(&parser, NOT, NULL, &state);
		    goto just_had_operator;
		}
		if (op == "XOR") {
		    Parse(&parser, XOR, NULL, &state);
		    goto just_had_operator;
		}
		if (op == "ADJ") {
		    if (it != end && *it == '/') {
			size_t width = 0;
			Utf8Iterator p = it;
			while (++p != end && U_isdigit(*p)) {
			    width = (width * 10) + (*p - '0');
			}
			if (width && (p == end || is_whitespace(*p))) {
			    it = p;
			    Parse(&parser, ADJ, new Term(width), &state);
			    goto just_had_operator;
			}
		    } else {
			Parse(&parser, ADJ, NULL, &state);
			goto just_had_operator;
		    }
		}
		if (op == "SYN") {
		    Parse(&parser, SYN, NULL, &state);
		    goto just_had_operator;
		}
	    } else if (op.size() == 2) {
		if (op == "OR") {
		    Parse(&parser, OR, NULL, &state);
		    goto just_had_operator;
		}
	    } else if (op.size() == 4) {
		if (op == "NEAR") {
		    if (it != end && *it == '/') {
			size_t width = 0;
			Utf8Iterator p = it;
			while (++p != end && U_isdigit(*p)) {
			    width = (width * 10) + (*p - '0');
			}
			if (width && (p == end || is_whitespace(*p))) {
			    it = p;
			    Parse(&parser, NEAR, new Term(width), &state);
			    goto just_had_operator;
			}
		    } else {
			Parse(&parser, NEAR, NULL, &state);
			goto just_had_operator;
		    }
		}
	    }
	}

	// If no prefix is set, use the default one.
	if (!field_info) field_info = prefix_stack.back();

	Assert(field_info->type == NON_BOOLEAN);

	{
	    string unstemmed_term(term);
	    term = Unicode::tolower(term);

	    // Reuse stem_strategy - STEM_SOME here means "stem terms except
	    // when used with positional operators".
	    stem_strategy stem_term = stem_action;
	    if (stem_term != STEM_NONE) {
		if (stemmer.is_none()) {
		    stem_term = STEM_NONE;
		} else if (first_wildcard != string::npos ||
			   edit_distance != NO_EDIT_DISTANCE) {
		    stem_term = STEM_NONE;
		} else if (stem_term == STEM_SOME ||
			   stem_term == STEM_SOME_FULL_POS) {
		    if (!should_stem(unstemmed_term, state)) {
			// E.g. don't stem `Tony` or `Keating`.
			stem_term = STEM_NONE;
		    } else if (it != end && is_stem_preventer(*it)) {
			// E.g. don't stem `tony` in `tony@example.org`.
			stem_term = STEM_NONE;
		    }
		}
	    }

	    if (first_wildcard != string::npos) {
		if (first_wildcard < state.get_min_wildcard_prefix_len()) {
		    errmsg = "Too few characters before wildcard";
		    return state.query;
		}
	    }

	    Term * term_obj = new Term(&state, term, field_info,
				       unstemmed_term, stem_term, term_pos++,
				       edit_distance);

	    if (first_wildcard != string::npos ||
		edit_distance != NO_EDIT_DISTANCE) {
		if (mode == IN_GROUP || mode == IN_GROUP2) {
		    // Drop out of IN_GROUP and flag that the group
		    // can be empty if all members are stopwords.
		    if (mode == IN_GROUP2)
			Parse(&parser, EMPTY_GROUP_OK, NULL, &state);
		    mode = DEFAULT;
		}
		Parse(&parser,
		      first_wildcard != string::npos ? WILD_TERM : EDIT_TERM,
		      term_obj,
		      &state);
		continue;
	    }

	    if (needs_word_break) {
		Parse(&parser, UNBROKEN_WORDS, term_obj, &state);
		// Drop out of IN_GROUP mode.
		if (mode == IN_GROUP || mode == IN_GROUP2)
		    mode = DEFAULT;
		if (it == end) break;
		continue;
	    }

	    if (mode == DEFAULT || mode == IN_GROUP || mode == IN_GROUP2) {
		if (it == end && (flags & FLAG_PARTIAL)) {
		    auto min_len = state.get_min_partial_prefix_len();
		    if (term_char_count >= min_len) {
			if (mode == IN_GROUP || mode == IN_GROUP2) {
			    // Drop out of IN_GROUP and flag that the group
			    // can be empty if all members are stopwords.
			    if (mode == IN_GROUP2)
				Parse(&parser, EMPTY_GROUP_OK, NULL, &state);
			    mode = DEFAULT;
			}
			// Final term of a partial match query, with no
			// following characters - treat as a wildcard.
			Parse(&parser, PARTIAL_TERM, term_obj, &state);
			continue;
		    }
		}
	    }

	    // Check spelling, if we're a normal term, and any of the prefixes
	    // are empty.
	    if ((flags & FLAG_SPELLING_CORRECTION) && !was_acronym) {
		const auto& prefixes = field_info->prefixes;
		for (const string& prefix : prefixes) {
		    if (!prefix.empty())
			continue;
		    const string & suggest = db.get_spelling_suggestion(term);
		    if (!suggest.empty()) {
			if (corrected_query.empty()) corrected_query = qs;
			size_t term_end_index = it.raw() - qs.data();
			size_t n = term_end_index - term_start_index;
			size_t pos = UNSIGNED_OVERFLOW_OK(term_start_index + correction_offset);
			corrected_query.replace(pos, n, suggest);
			UNSIGNED_OVERFLOW_OK(correction_offset += suggest.size());
			UNSIGNED_OVERFLOW_OK(correction_offset -= n);
		    }
		    break;
		}
	    }

	    if (mode == IN_PHRASED_TERM) {
		Parse(&parser, PHR_TERM, term_obj, &state);
	    } else {
		// See if the next token will be PHR_TERM - if so, this one
		// needs to be TERM not GROUP_TERM.
		if ((mode == IN_GROUP || mode == IN_GROUP2) &&
		    is_phrase_generator(*it)) {
		    // FIXME: can we clean this up?
		    Utf8Iterator p = it;
		    do {
			++p;
		    } while (p != end && is_phrase_generator(*p));
		    // Don't generate a phrase unless the phrase generators are
		    // immediately followed by another term.
		    if (p != end && is_wordchar(*p)) {
			mode = DEFAULT;
		    }
		}

		int token = TERM;
		if (mode == IN_GROUP || mode == IN_GROUP2) {
		    mode = IN_GROUP2;
		    token = GROUP_TERM;
		}
		Parse(&parser, token, term_obj, &state);
		if (token == TERM && mode != DEFAULT)
		    continue;
	    }
	}

	if (it == end) break;

	if (is_phrase_generator(*it)) {
	    // Skip multiple phrase generators.
	    do {
		++it;
	    } while (it != end && is_phrase_generator(*it));
	    // Don't generate a phrase unless the phrase generators are
	    // immediately followed by another term.
	    if (it != end && is_wordchar(*it)) {
		mode = IN_PHRASED_TERM;
		term_start_index = it.raw() - qs.data();
		goto phrased_term;
	    }
	} else if (mode == DEFAULT || mode == IN_GROUP || mode == IN_GROUP2) {
	    int old_mode = mode;
	    mode = DEFAULT;
	    if (!last_was_operator_needing_term && is_whitespace(*it)) {
		newprev = ' ';
		// Skip multiple whitespace.
		do {
		    ++it;
		} while (it != end && is_whitespace(*it));
		// Don't generate a group unless the terms are only separated
		// by whitespace.
		if (it != end && is_wordchar(*it)) {
		    if (old_mode == IN_GROUP || old_mode == IN_GROUP2) {
			mode = IN_GROUP2;
		    } else {
			mode = IN_GROUP;
		    }
		}
	    }
	}
    }
done:
    if (!state.error) {
	// Implicitly close any unclosed quotes.
	if (mode == IN_QUOTES || mode == IN_PREFIXED_QUOTES)
	    Parse(&parser, QUOTE, NULL, &state);

	// Implicitly close all unclosed brackets.
	while (prefix_stack.size() > 1) {
	    Parse(&parser, KET, NULL, &state);
	    prefix_stack.pop_back();
	}
	Parse(&parser, 0, NULL, &state);
    }

    errmsg = state.error;
    return state.query;
}

#line 4049 "queryparser/queryparser_internal.cc"
