/** @file
 * @brief tests which don't need a backend
 */
/* Copyright (C) 2009 Richard Boulton
 * Copyright (C) 2009-2026 Olly Betts
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include <config.h>

#include "api_none.h"

#define XAPIAN_DEPRECATED(D) D
#include <xapian.h>

#include "apitest.h"
#include "str.h"
#include "testsuite.h"
#include "testutils.h"

#include <string_view>
#include <vector>

using namespace std;

// Check the version functions give consistent results.
DEFINE_TESTCASE(version1, !backend) {
    string version = str(Xapian::major_version());
    version += '.';
    version += str(Xapian::minor_version());
    version += '.';
    version += str(Xapian::revision());
    TEST_EQUAL(Xapian::version_string(), version);
}

// Regression test: various methods on Database() used to segfault or cause
// division by 0.  Fixed in 1.1.4 and 1.0.18.  Ticket#415.
DEFINE_TESTCASE(nosubdatabases1, !backend) {
    Xapian::Database db;
    TEST(db.get_metadata("foo").empty());
    TEST_EQUAL(db.metadata_keys_begin(), db.metadata_keys_end());
    TEST_EXCEPTION(Xapian::InvalidOperationError, db.termlist_begin(1));
    TEST_EQUAL(db.allterms_begin(), db.allterms_end());
    TEST_EQUAL(db.allterms_begin("foo"), db.allterms_end("foo"));
    TEST_EXCEPTION(Xapian::InvalidOperationError, db.positionlist_begin(1, "foo"));
    TEST_EQUAL(db.get_lastdocid(), 0);
    TEST_EQUAL(db.valuestream_begin(7), db.valuestream_end(7));
    TEST_EXCEPTION(Xapian::InvalidOperationError, db.get_doclength(1));
    TEST_EXCEPTION(Xapian::InvalidOperationError, db.get_unique_terms(1));
    TEST_EXCEPTION(Xapian::InvalidOperationError, db.get_document(1));

    Xapian::WritableDatabase wdb;
    TEST_EXCEPTION(Xapian::InvalidOperationError, wdb.begin_transaction());
    TEST_EXCEPTION(Xapian::InvalidOperationError, wdb.commit_transaction());
    TEST_EXCEPTION(Xapian::InvalidOperationError, wdb.cancel_transaction());
}

/// Feature test for Document::add_boolean_term(), new in 1.0.18/1.1.4.
DEFINE_TESTCASE(document1, !backend) {
    Xapian::Document doc;
    doc.add_boolean_term("Hxapian.org");
    TEST_EQUAL(doc.termlist_count(), 1);
    Xapian::TermIterator t = doc.termlist_begin();
    TEST(t != doc.termlist_end());
    TEST_EQUAL(*t, "Hxapian.org");
    TEST_EQUAL(t.get_wdf(), 0);
    TEST(++t == doc.termlist_end());
    doc.remove_term("Hxapian.org");
    TEST_EQUAL(doc.termlist_count(), 0);
    TEST(doc.termlist_begin() == doc.termlist_end());
}

/// Regression test - the docid wasn't initialised prior to 1.0.22/1.2.4.
DEFINE_TESTCASE(document2, !backend) {
    Xapian::Document doc;
    // The return value is uninitialised, so running under valgrind this
    // will fail reliably prior to the fix.
    TEST_EQUAL(doc.get_docid(), 0);
}

/// Feature tests for Document::clear_terms().
DEFINE_TESTCASE(documentclearterms1, !backend) {
    {
	Xapian::Document doc;
	doc.add_boolean_term("Hlocalhost");
	doc.add_term("hello");
	doc.add_term("there", 2);
	doc.add_posting("positional", 1);
	doc.add_posting("information", 2, 3);
	TEST_EQUAL(doc.termlist_count(), 5);
	TEST(doc.termlist_begin() != doc.termlist_end());
	doc.clear_terms();
	TEST_EQUAL(doc.termlist_count(), 0);
	TEST(doc.termlist_begin() == doc.termlist_end());
	// Test clear_terms() when there are no terms.
	doc.clear_terms();
	TEST_EQUAL(doc.termlist_count(), 0);
	TEST(doc.termlist_begin() == doc.termlist_end());
    }

    {
	// Test clear_terms() when there have never been any terms.
	Xapian::Document doc;
	doc.clear_terms();
	TEST_EQUAL(doc.termlist_count(), 0);
	TEST(doc.termlist_begin() == doc.termlist_end());
    }
}

/// Feature tests for Document::clear_values().
DEFINE_TESTCASE(documentclearvalues1, !backend) {
    {
	Xapian::Document doc;
	doc.add_value(37, "hello");
	doc.add_value(42, "world");
	TEST_EQUAL(doc.values_count(), 2);
	TEST(doc.values_begin() != doc.values_end());
	doc.clear_values();
	TEST_EQUAL(doc.values_count(), 0);
	TEST(doc.values_begin() == doc.values_end());
	// Test clear_values() when there are no values.
	doc.clear_values();
	TEST_EQUAL(doc.values_count(), 0);
	TEST(doc.values_begin() == doc.values_end());
    }

    {
	// Test clear_values() when there have never been any values.
	Xapian::Document doc;
	doc.clear_values();
	TEST_EQUAL(doc.values_count(), 0);
	TEST(doc.termlist_begin() == doc.termlist_end());
    }
}

/// Feature tests for errors for empty terms.
DEFINE_TESTCASE(documentemptyterm1, !backend) {
    Xapian::Document doc;
    TEST_EXCEPTION(Xapian::InvalidArgumentError,
	    doc.add_boolean_term(string()));
    TEST_EXCEPTION(Xapian::InvalidArgumentError,
	    doc.add_term(string()));
    TEST_EXCEPTION(Xapian::InvalidArgumentError,
	    doc.add_posting(string(), 1));
    TEST_EXCEPTION(Xapian::InvalidArgumentError,
	    doc.add_posting(string(), 2, 3));
    TEST_EXCEPTION(Xapian::InvalidArgumentError,
	    doc.remove_term(string()));
    TEST_EXCEPTION(Xapian::InvalidArgumentError,
	    doc.remove_posting(string(), 1));
    TEST_EXCEPTION(Xapian::InvalidArgumentError,
	    doc.remove_posting(string(), 2, 3));
    TEST_EXCEPTION(Xapian::InvalidArgumentError,
	    doc.remove_postings(string(), 2, 3));
    TEST_EXCEPTION(Xapian::InvalidArgumentError,
	    doc.remove_postings(string(), 2, 3, 4));
}

DEFINE_TESTCASE(emptyquery4, !backend) {
    // Test we get an empty query from applying any of the following ops to
    // an empty list of subqueries.
    Xapian::Query q;
    TEST(Xapian::Query(q.OP_AND, &q, &q).empty());
    TEST(Xapian::Query(q.OP_OR, &q, &q).empty());
    TEST(Xapian::Query(q.OP_AND_NOT, &q, &q).empty());
    TEST(Xapian::Query(q.OP_XOR, &q, &q).empty());
    TEST(Xapian::Query(q.OP_AND_MAYBE, &q, &q).empty());
    TEST(Xapian::Query(q.OP_FILTER, &q, &q).empty());
    TEST(Xapian::Query(q.OP_NEAR, &q, &q).empty());
    TEST(Xapian::Query(q.OP_PHRASE, &q, &q).empty());
    TEST(Xapian::Query(q.OP_ELITE_SET, &q, &q).empty());
    TEST(Xapian::Query(q.OP_SYNONYM, &q, &q).empty());
    TEST(Xapian::Query(q.OP_MAX, &q, &q).empty());
}

DEFINE_TESTCASE(singlesubquery1, !backend) {
    // Test that we get just the subquery if we apply any of the following
    // ops to just that subquery.
#define singlesubquery1_(OP) \
    TEST_STRINGS_EQUAL(Xapian::Query(q->OP, q, q + 1).get_description(),\
	"Query(test)")
    Xapian::Query q[1] = { Xapian::Query("test") };
    singlesubquery1_(OP_AND);
    singlesubquery1_(OP_OR);
    singlesubquery1_(OP_AND_NOT);
    singlesubquery1_(OP_XOR);
    singlesubquery1_(OP_AND_MAYBE);
    singlesubquery1_(OP_FILTER);
    singlesubquery1_(OP_NEAR);
    singlesubquery1_(OP_PHRASE);
    singlesubquery1_(OP_ELITE_SET);
    singlesubquery1_(OP_SYNONYM);
    singlesubquery1_(OP_MAX);
}

DEFINE_TESTCASE(singlesubquery2, !backend) {
    // Like the previous test, but using MatchNothing as the subquery.
#define singlesubquery2_(OP) \
    TEST_STRINGS_EQUAL(Xapian::Query(q->OP, q, q + 1).get_description(),\
	"Query()")
    Xapian::Query q[1] = { Xapian::Query::MatchNothing };
    singlesubquery2_(OP_AND);
    singlesubquery2_(OP_OR);
    singlesubquery2_(OP_AND_NOT);
    singlesubquery2_(OP_XOR);
    singlesubquery2_(OP_AND_MAYBE);
    singlesubquery2_(OP_FILTER);
    singlesubquery2_(OP_NEAR);
    singlesubquery2_(OP_PHRASE);
    singlesubquery2_(OP_ELITE_SET);
    singlesubquery2_(OP_SYNONYM);
    singlesubquery2_(OP_MAX);
}

DEFINE_TESTCASE(singlesubquery3, !backend) {
    // Like the previous test, but using MatchAll as the subquery.
#define singlesubquery3_(OP) \
    TEST_STRINGS_EQUAL(Xapian::Query(q->OP, q, q + 1).get_description(),\
	"Query(<alldocuments>)")
    Xapian::Query q[1] = { Xapian::Query::MatchAll };
    singlesubquery3_(OP_AND);
    singlesubquery3_(OP_OR);
    singlesubquery3_(OP_AND_NOT);
    singlesubquery3_(OP_XOR);
    singlesubquery3_(OP_AND_MAYBE);
    singlesubquery3_(OP_FILTER);
    // OP_NEAR and OP_PHRASE over MatchAll doesn't really make sense.
    singlesubquery3_(OP_ELITE_SET);
    singlesubquery3_(OP_SYNONYM);
    singlesubquery3_(OP_MAX);
}

DEFINE_TESTCASE(pairwisequery1, !backend) {
    // Test that constructing from char* and const char* gives the same result
    // as explicitly constructing Query objects for the subqueries.  This
    // serves as a regression test for using char*/const char* not compiling
    // with Xapian <= 1.4.30.
#define lit_a "a"
#define lit_b "b"
    char aa[] = lit_a;
    char ab[] = lit_b;
    const char* ca = aa;
    const char* cb = ab;
    Xapian::Query qa(aa);
    Xapian::Query qb(ab);

#define pairwisequery1_(OP) do {\
	auto expect = Xapian::Query(qa.OP, lit_a, lit_b).get_description(); \
	TEST_STRINGS_EQUAL(expect, \
			   Xapian::Query(qa.OP, aa, ab).get_description()); \
	TEST_STRINGS_EQUAL(expect, \
			   Xapian::Query(qa.OP, ca, cb).get_description()); \
	TEST_STRINGS_EQUAL(expect, \
			   Xapian::Query(qa.OP, qa, qb).get_description()); \
    } while (false)

    pairwisequery1_(OP_AND);
    pairwisequery1_(OP_OR);
    pairwisequery1_(OP_AND_NOT);
    pairwisequery1_(OP_XOR);
    pairwisequery1_(OP_AND_MAYBE);
    pairwisequery1_(OP_FILTER);
    pairwisequery1_(OP_NEAR);
    pairwisequery1_(OP_PHRASE);
    pairwisequery1_(OP_ELITE_SET);
    pairwisequery1_(OP_SYNONYM);
    pairwisequery1_(OP_MAX);
}

/// Check we no longer combine wqf for same term at the same position.
DEFINE_TESTCASE(combinewqfnomore1, !backend) {
    Xapian::Query q(Xapian::Query::OP_OR,
		    Xapian::Query("beer", 1, 1),
		    Xapian::Query("beer", 1, 1));
    // Prior to 1.3.0, we would have given beer@2, but we decided that wasn't
    // really useful or helpful.
    TEST_EQUAL(q.get_description(), "Query((beer@1 OR beer@1))");
}

class DestroyedFlag {
    bool & destroyed;

  public:
    DestroyedFlag(bool & destroyed_) : destroyed(destroyed_) {
	destroyed = false;
    }

    ~DestroyedFlag() {
	destroyed = true;
    }
};

class TestRangeProcessor : public Xapian::RangeProcessor {
    DestroyedFlag destroyed;

  public:
    TestRangeProcessor(bool & destroyed_)
	: Xapian::RangeProcessor(0), destroyed(destroyed_) { }

    Xapian::Query operator()(const std::string&, const std::string&) override {
	return Xapian::Query::MatchAll;
    }
};

/// Check reference counting of user-subclassable classes.
DEFINE_TESTCASE(subclassablerefcount1, !backend) {
    bool gone_auto, gone;
#ifdef _MSC_VER
    // MSVC incorrectly warns these are potentially uninitialised.  It's
    // unhelpful to always initialise these as that could mask if a genuine bug
    // were introduced (which currently would likely be caught by a warning
    // from a smarter compiler).
    gone_auto = gone = false;
#endif

    // Simple test of release().
    {
	Xapian::RangeProcessor * rp = new TestRangeProcessor(gone);
	TEST(!gone);
	Xapian::QueryParser qp;
	qp.add_rangeprocessor(rp->release());
	TEST(!gone);
    }
    TEST(gone);

    // Check a second call to release() has no effect.
    {
	Xapian::RangeProcessor * rp = new TestRangeProcessor(gone);
	TEST(!gone);
	Xapian::QueryParser qp;
	qp.add_rangeprocessor(rp->release());
	rp->release();
	TEST(!gone);
    }
    TEST(gone);

    // Test reference counting works, and that a RangeProcessor with automatic
    // storage works OK.
    {
	TestRangeProcessor rp_auto(gone_auto);
	TEST(!gone_auto);
	{
	    Xapian::QueryParser qp1;
	    {
		Xapian::QueryParser qp2;
		Xapian::RangeProcessor * rp;
		rp = new TestRangeProcessor(gone);
		TEST(!gone);
		qp1.add_rangeprocessor(rp->release());
		TEST(!gone);
		qp2.add_rangeprocessor(rp);
		TEST(!gone);
		qp2.add_rangeprocessor(&rp_auto);
		TEST(!gone);
		TEST(!gone_auto);
	    }
	    TEST(!gone);
	}
	TEST(gone);
	TEST(!gone_auto);
    }
    TEST(gone_auto);

    // Regression test for initial implementation, where ~opt_intrusive_ptr()
    // checked the reference of the object, which may have already been deleted
    // if it wasn't been reference counted.
    {
	Xapian::QueryParser qp;
	{
	    Xapian::RangeProcessor * rp = new TestRangeProcessor(gone);
	    TEST(!gone);
	    qp.add_rangeprocessor(rp);
	    delete rp;
	    TEST(gone);
	}
	// At the end of this block, qp is destroyed, but mustn't dereference
	// the pointer it has to rp.  If it does, that should get caught
	// when tests are run under valgrind.
    }
}

class TestFieldProcessor : public Xapian::FieldProcessor {
    DestroyedFlag destroyed;

  public:
    TestFieldProcessor(bool & destroyed_) : destroyed(destroyed_) { }

    Xapian::Query operator()(const string& str) override {
	return Xapian::Query(str);
    }
};

/// Check reference counting of user-subclassable classes.
DEFINE_TESTCASE(subclassablerefcount2, !backend) {
    bool gone_auto, gone;
#ifdef _MSC_VER
    // MSVC incorrectly warns these are potentially uninitialised.  It's
    // unhelpful to always initialise these as that could mask if a genuine bug
    // were introduced (which currently would likely be caught by a warning
    // from a smarter compiler).
    gone_auto = gone = false;
#endif

    // Simple test of release().
    {
	Xapian::FieldProcessor * proc = new TestFieldProcessor(gone);
	TEST(!gone);
	Xapian::QueryParser qp;
	qp.add_prefix("foo", proc->release());
	TEST(!gone);
    }
    TEST(gone);

    // Check a second call to release() has no effect.
    {
	Xapian::FieldProcessor * proc = new TestFieldProcessor(gone);
	TEST(!gone);
	Xapian::QueryParser qp;
	qp.add_prefix("foo", proc->release());
	proc->release();
	TEST(!gone);
    }
    TEST(gone);

    // Test reference counting works, and that a FieldProcessor with automatic
    // storage works OK.
    {
	TestFieldProcessor proc_auto(gone_auto);
	TEST(!gone_auto);
	{
	    Xapian::QueryParser qp1;
	    {
		Xapian::QueryParser qp2;
		Xapian::FieldProcessor * proc;
		proc = new TestFieldProcessor(gone);
		TEST(!gone);
		qp1.add_prefix("foo", proc->release());
		TEST(!gone);
		qp2.add_prefix("foo", proc);
		TEST(!gone);
		qp2.add_prefix("bar", &proc_auto);
		TEST(!gone);
		TEST(!gone_auto);
	    }
	    TEST(!gone);
	}
	TEST(gone);
	TEST(!gone_auto);
    }
    TEST(gone_auto);
}

class TestMatchSpy : public Xapian::MatchSpy {
    DestroyedFlag destroyed;

  public:
    TestMatchSpy(bool & destroyed_) : destroyed(destroyed_) { }

    void operator()(const Xapian::Document&, double) override { }
};

/// Check reference counting of MatchSpy.
DEFINE_TESTCASE(subclassablerefcount3, backend) {
    Xapian::Database db = get_database("apitest_simpledata");

    bool gone_auto, gone;
#ifdef _MSC_VER
    // MSVC incorrectly warns these are potentially uninitialised.  It's
    // unhelpful to always initialise these as that could mask if a genuine bug
    // were introduced (which currently would likely be caught by a warning
    // from a smarter compiler).
    gone_auto = gone = false;
#endif

    // Simple test of release().
    {
	Xapian::MatchSpy * spy = new TestMatchSpy(gone);
	TEST(!gone);
	Xapian::Enquire enquire(db);
	enquire.add_matchspy(spy->release());
	TEST(!gone);
    }
    TEST(gone);

    // Check a second call to release() has no effect.
    {
	Xapian::MatchSpy * spy = new TestMatchSpy(gone);
	TEST(!gone);
	Xapian::Enquire enquire(db);
	enquire.add_matchspy(spy->release());
	spy->release();
	TEST(!gone);
    }
    TEST(gone);

    // Test reference counting works, and that a MatchSpy with automatic
    // storage works OK.
    {
	TestMatchSpy spy_auto(gone_auto);
	TEST(!gone_auto);
	{
	    Xapian::Enquire enq1(db);
	    {
		Xapian::Enquire enq2(db);
		Xapian::MatchSpy * spy;
		spy = new TestMatchSpy(gone);
		TEST(!gone);
		enq1.add_matchspy(spy->release());
		TEST(!gone);
		enq2.add_matchspy(spy);
		TEST(!gone);
		enq2.add_matchspy(&spy_auto);
		TEST(!gone);
		TEST(!gone_auto);
	    }
	    TEST(!gone);
	}
	TEST(gone);
	TEST(!gone_auto);
    }
    TEST(gone_auto);
}

class TestStopper : public Xapian::Stopper {
    DestroyedFlag destroyed;

  public:
    TestStopper(bool & destroyed_) : destroyed(destroyed_) { }

    bool operator()(const std::string&) const override { return true; }
};

/// Check reference counting of Stopper with QueryParser.
DEFINE_TESTCASE(subclassablerefcount4, !backend) {
    bool gone_auto, gone;
#ifdef _MSC_VER
    // MSVC incorrectly warns these are potentially uninitialised.  It's
    // unhelpful to always initialise these as that could mask if a genuine bug
    // were introduced (which currently would likely be caught by a warning
    // from a smarter compiler).
    gone_auto = gone = false;
#endif

    // Simple test of release().
    {
	Xapian::Stopper * stopper = new TestStopper(gone);
	TEST(!gone);
	Xapian::QueryParser qp;
	qp.set_stopper(stopper->release());
	TEST(!gone);
    }
    TEST(gone);

    // Test that setting a new stopper causes the previous one to be released.
    {
	bool gone0;
	Xapian::Stopper * stopper0 = new TestStopper(gone0);
	TEST(!gone0);
	Xapian::QueryParser qp;
	qp.set_stopper(stopper0->release());
	TEST(!gone0);

	Xapian::Stopper * stopper = new TestStopper(gone);
	TEST(!gone);
	qp.set_stopper(stopper->release());
	TEST(gone0);
	TEST(!gone);
    }
    TEST(gone);

    // Check a second call to release() has no effect.
    {
	Xapian::Stopper * stopper = new TestStopper(gone);
	TEST(!gone);
	Xapian::QueryParser qp;
	qp.set_stopper(stopper->release());
	stopper->release();
	TEST(!gone);
    }
    TEST(gone);

    // Test reference counting works, and that a Stopper with automatic
    // storage works OK.
    {
	TestStopper stopper_auto(gone_auto);
	TEST(!gone_auto);
	{
	    Xapian::QueryParser qp1;
	    {
		Xapian::QueryParser qp2;
		Xapian::Stopper * stopper;
		stopper = new TestStopper(gone);
		TEST(!gone);
		qp1.set_stopper(stopper->release());
		TEST(!gone);
		qp2.set_stopper(stopper);
		TEST(!gone);
		qp2.set_stopper(&stopper_auto);
		TEST(!gone);
		TEST(!gone_auto);
	    }
	    TEST(!gone);
	}
	TEST(gone);
	TEST(!gone_auto);
    }
    TEST(gone_auto);
}

/// Check reference counting of Stopper with TermGenerator.
DEFINE_TESTCASE(subclassablerefcount5, !backend) {
    bool gone_auto, gone;
#ifdef _MSC_VER
    // MSVC incorrectly warns these are potentially uninitialised.  It's
    // unhelpful to always initialise these as that could mask if a genuine bug
    // were introduced (which currently would likely be caught by a warning
    // from a smarter compiler).
    gone_auto = gone = false;
#endif

    // Simple test of release().
    {
	Xapian::Stopper * stopper = new TestStopper(gone);
	TEST(!gone);
	Xapian::TermGenerator indexer;
	indexer.set_stopper(stopper->release());
	TEST(!gone);
    }
    TEST(gone);

    // Test that setting a new stopper causes the previous one to be released.
    {
	bool gone0;
	Xapian::Stopper * stopper0 = new TestStopper(gone0);
	TEST(!gone0);
	Xapian::TermGenerator indexer;
	indexer.set_stopper(stopper0->release());
	TEST(!gone0);

	Xapian::Stopper * stopper = new TestStopper(gone);
	TEST(!gone);
	indexer.set_stopper(stopper->release());
	TEST(gone0);
	TEST(!gone);
    }
    TEST(gone);

    // Check a second call to release() has no effect.
    {
	Xapian::Stopper * stopper = new TestStopper(gone);
	TEST(!gone);
	Xapian::TermGenerator indexer;
	indexer.set_stopper(stopper->release());
	stopper->release();
	TEST(!gone);
    }
    TEST(gone);

    // Test reference counting works, and that a Stopper with automatic
    // storage works OK.
    {
	TestStopper stopper_auto(gone_auto);
	TEST(!gone_auto);
	{
	    Xapian::TermGenerator indexer1;
	    {
		Xapian::TermGenerator indexer2;
		Xapian::Stopper * stopper;
		stopper = new TestStopper(gone);
		TEST(!gone);
		indexer1.set_stopper(stopper->release());
		TEST(!gone);
		indexer2.set_stopper(stopper);
		TEST(!gone);
		indexer2.set_stopper(&stopper_auto);
		TEST(!gone);
		TEST(!gone_auto);
	    }
	    TEST(!gone);
	}
	TEST(gone);
	TEST(!gone_auto);
    }
    TEST(gone_auto);
}

class TestKeyMaker : public Xapian::KeyMaker {
    DestroyedFlag destroyed;

  public:
    TestKeyMaker(bool & destroyed_) : destroyed(destroyed_) { }

    string operator()(const Xapian::Document&) const override {
	return string();
    }
};

/// Check reference counting of KeyMaker.
DEFINE_TESTCASE(subclassablerefcount6, backend) {
    Xapian::Database db = get_database("apitest_simpledata");

    bool gone_auto, gone;
#ifdef _MSC_VER
    // MSVC incorrectly warns these are potentially uninitialised.  It's
    // unhelpful to always initialise these as that could mask if a genuine bug
    // were introduced (which currently would likely be caught by a warning
    // from a smarter compiler).
    gone_auto = gone = false;
#endif

    // Simple test of release().
    {
	Xapian::KeyMaker * keymaker = new TestKeyMaker(gone);
	TEST(!gone);
	Xapian::Enquire enq(db);
	enq.set_sort_by_key(keymaker->release(), false);
	TEST(!gone);
    }
    TEST(gone);

    // Test that setting a new keymaker causes the previous one to be released.
    {
	bool gone0;
	Xapian::KeyMaker * keymaker0 = new TestKeyMaker(gone0);
	TEST(!gone0);
	Xapian::Enquire enq(db);
	enq.set_sort_by_key(keymaker0->release(), false);
	TEST(!gone0);

	Xapian::KeyMaker * keymaker = new TestKeyMaker(gone);
	TEST(!gone);
	enq.set_sort_by_key_then_relevance(keymaker->release(), false);
	TEST(gone0);
	TEST(!gone);
    }
    TEST(gone);

    // Check a second call to release() has no effect.
    {
	Xapian::KeyMaker * keymaker = new TestKeyMaker(gone);
	TEST(!gone);
	Xapian::Enquire enq(db);
	enq.set_sort_by_key(keymaker->release(), false);
	keymaker->release();
	TEST(!gone);
    }
    TEST(gone);

    // Test reference counting works, and that a KeyMaker with automatic
    // storage works OK.
    {
	TestKeyMaker keymaker_auto(gone_auto);
	TEST(!gone_auto);
	{
	    Xapian::Enquire enq1(db);
	    {
		Xapian::Enquire enq2(db);
		Xapian::KeyMaker * keymaker;
		keymaker = new TestKeyMaker(gone);
		TEST(!gone);
		enq1.set_sort_by_key(keymaker->release(), false);
		TEST(!gone);
		enq2.set_sort_by_relevance_then_key(keymaker, false);
		TEST(!gone);
		enq2.set_sort_by_key_then_relevance(&keymaker_auto, false);
		TEST(!gone);
		TEST(!gone_auto);
	    }
	    TEST(!gone);
	}
	TEST(gone);
	TEST(!gone_auto);
    }
    TEST(gone_auto);
}

class TestExpandDecider : public Xapian::ExpandDecider {
    DestroyedFlag destroyed;

  public:
    TestExpandDecider(bool & destroyed_) : destroyed(destroyed_) { }

    bool operator()(const string&) const override { return true; }
};

/// Check reference counting of ExpandDecider.
DEFINE_TESTCASE(subclassablerefcount7, backend) {
    Xapian::Database db = get_database("apitest_simpledata");
    Xapian::Enquire enq(db);
    Xapian::RSet rset;
    rset.add_document(1);

    bool gone_auto, gone;
#ifdef _MSC_VER
    // MSVC incorrectly warns these are potentially uninitialised.  It's
    // unhelpful to always initialise these as that could mask if a genuine bug
    // were introduced (which currently would likely be caught by a warning
    // from a smarter compiler).
    gone_auto = gone = false;
#endif

    for (int flags = 0;
	 flags <= Xapian::Enquire::INCLUDE_QUERY_TERMS;
	 flags += Xapian::Enquire::INCLUDE_QUERY_TERMS) {
	// Test of auto lifetime ExpandDecider.
	{
	    TestExpandDecider edecider_auto(gone_auto);
	    TEST(!gone_auto);
	    (void)enq.get_eset(5, rset, 0, &edecider_auto);
	    TEST(!gone_auto);
	}
	TEST(gone_auto);

	// Simple test of release().
	{
	    Xapian::ExpandDecider * edecider = new TestExpandDecider(gone);
	    TEST(!gone);
	    (void)enq.get_eset(5, rset, 0, edecider);
	    TEST(!gone);
	    delete edecider;
	    TEST(gone);
	}

	// Test that a released ExpandDecider gets cleaned up by get_eset().
	{
	    Xapian::ExpandDecider * edecider = new TestExpandDecider(gone);
	    TEST(!gone);
	    (void)enq.get_eset(5, rset, 0, edecider->release());
	    TEST(gone);
	}

	// Check a second call to release() has no effect.
	{
	    Xapian::ExpandDecider * edecider = new TestExpandDecider(gone);
	    TEST(!gone);
	    edecider->release();
	    TEST(!gone);
	    (void)enq.get_eset(5, rset, 0, edecider->release());
	    TEST(gone);
	}
    }

    // Test combinations of released/non-released with ExpandDeciderAnd.
    {
	TestExpandDecider edecider_auto(gone_auto);
	TEST(!gone_auto);
	Xapian::ExpandDecider * edecider = new TestExpandDecider(gone);
	TEST(!gone);
	(void)enq.get_eset(5, rset, 0,
		(new Xapian::ExpandDeciderAnd(
		    &edecider_auto,
		    edecider->release()))->release());
	TEST(!gone_auto);
	TEST(gone);
    }
    TEST(gone_auto);
    {
	TestExpandDecider edecider_auto(gone_auto);
	TEST(!gone_auto);
	Xapian::ExpandDecider * edecider = new TestExpandDecider(gone);
	TEST(!gone);
	(void)enq.get_eset(5, rset, 0,
		(new Xapian::ExpandDeciderAnd(
		    edecider->release(),
		    &edecider_auto))->release());
	TEST(!gone_auto);
	TEST(gone);
    }
    TEST(gone_auto);
}

/// Check encoding of non-UTF8 document data.
DEFINE_TESTCASE(nonutf8docdesc1, !backend) {
    Xapian::Document doc;
    doc.set_data("\xc0\x80\xf5\x80\x80\x80\xfe\xff");
    TEST_EQUAL(doc.get_description(),
	      "Document(docid=0, data=\\xc0\\x80\\xf5\\x80\\x80\\x80\\xfe\\xff)");
    doc.set_data(string("\x00\x1f", 2));
    TEST_EQUAL(doc.get_description(),
	      "Document(docid=0, data=\\x00\\x1f)");
    // Check that backslashes are encoded so output isn't ambiguous.
    doc.set_data("back\\slash");
    TEST_EQUAL(doc.get_description(),
	      "Document(docid=0, data=back\\x5cslash)");
}

DEFINE_TESTCASE(orphaneddoctermitor1, !backend) {
    Xapian::TermIterator t;
    {
	Xapian::Document doc;
	doc.add_term("foo");
	t = doc.termlist_begin();
    }
    TEST_EQUAL(*t, "foo");
}

/** Test removal of terms from a document while iterating over them.
 *
 *  Prior to 1.4.6 the underlying iterator was invalidated when
 *  preinc == false, leading to undefined behaviour (typically a segmentation
 *  fault).
 */
DEFINE_TESTCASE(deletewhileiterating1, !backend) {
    for (bool preinc : { false, true }) {
	Xapian::Document doc;
	Xapian::TermGenerator indexer;
	indexer.set_document(doc);
	indexer.index_text("Pull the rug out from under ourselves", 1, "S");
	Xapian::TermIterator term_iterator = doc.termlist_begin();
	term_iterator.skip_to("S");
	while (term_iterator != doc.termlist_end()) {
	    const string& term = *term_iterator;
	    if (!startswith(term, "S")) {
		break;
	    }
	    if (preinc) ++term_iterator;
	    doc.remove_term(term);
	    if (!preinc) ++term_iterator;
	}
	TEST_EQUAL(doc.termlist_count(), 0);
	TEST(doc.termlist_begin() == doc.termlist_end());
    }
}

/// Feature test for Document::remove_postings().
DEFINE_TESTCASE(removepostings, !backend) {
    Xapian::Document doc;
    // Add Fibonacci sequence as positions.
    Xapian::termpos prev_pos = 1;
    Xapian::termpos pos = 1;
    while (pos < 1000) {
	doc.add_posting("foo", pos);
	auto new_pos = prev_pos + pos;
	prev_pos = pos;
	pos = new_pos;
    }

    // Check we added exactly one term.
    TEST_EQUAL(doc.termlist_count(), 1);

    Xapian::TermIterator t = doc.termlist_begin();
    auto num_pos = t.positionlist_count();
    TEST_EQUAL(t.get_wdf(), num_pos);

    Xapian::PositionIterator pi = t.positionlist_begin();

    // Out of order is a no-op.
    TEST_EQUAL(doc.remove_postings("foo", 2, 1), 0);
    t = doc.termlist_begin();
    TEST_EQUAL(t.positionlist_count(), num_pos);
    TEST_EQUAL(t.get_wdf(), num_pos);

    // 6 and 7 aren't in the sequence.
    TEST_EQUAL(doc.remove_postings("foo", 6, 7), 0);
    t = doc.termlist_begin();
    TEST_EQUAL(t.positionlist_count(), num_pos);
    TEST_EQUAL(t.get_wdf(), num_pos);

    // Beyond the end of the positions.
    TEST_EQUAL(doc.remove_postings("foo", 1000, 2000), 0);
    t = doc.termlist_begin();
    TEST_EQUAL(t.positionlist_count(), num_pos);
    TEST_EQUAL(t.get_wdf(), num_pos);

    // 1, 2, 3 are in the sequence, 4 isn't.
    TEST_EQUAL(doc.remove_postings("foo", 1, 4), 3);
    t = doc.termlist_begin();
    TEST_EQUAL(t.positionlist_count(), num_pos - 3);
    TEST_EQUAL(t.get_wdf(), num_pos - 3);

    // Test an existing PositionIterator isn't affected.
    TEST_EQUAL(*pi, 1);
    TEST_EQUAL(*++pi, 2);
    TEST_EQUAL(*++pi, 3);
    TEST_EQUAL(*++pi, 5);

    // Remove the end position.
    TEST_EQUAL(doc.remove_postings("foo", 876, 987), 1);
    t = doc.termlist_begin();
    TEST_EQUAL(t.positionlist_count(), num_pos - 4);
    TEST_EQUAL(t.get_wdf(), num_pos - 4);

    // Remove a range in the middle.
    TEST_EQUAL(doc.remove_postings("foo", 33, 233), 5);
    t = doc.termlist_begin();
    TEST_EQUAL(t.positionlist_count(), num_pos - 9);
    TEST_EQUAL(t.get_wdf(), num_pos - 9);

    // Check the expected positions are left.
    t = doc.termlist_begin();
    static const Xapian::termpos expected[] = { 5, 8, 13, 21, 377, 610, 9999 };
    const Xapian::termpos* expect = expected;
    for (auto p = t.positionlist_begin(); p != t.positionlist_end(); ++p) {
	TEST_EQUAL(*p, *expect);
	++expect;
    }
    TEST_EQUAL(*expect, 9999);
    TEST_EQUAL(t.get_wdf(), 6);

    pi = t.positionlist_begin();

    // Test remove_position().
    doc.remove_posting("foo", 8);

    // Test an existing PositionIterator isn't affected.
    TEST_EQUAL(*pi, 5);
    TEST_EQUAL(*++pi, 8);

    // Check removing all positions removes the term too if the wdf reaches 0
    // (since 2.0.0).
    TEST_EQUAL(doc.remove_postings("foo", 5, 1000), 5);
    t = doc.termlist_begin();
    TEST(t == doc.termlist_end());

    // Test the removing all positions doesn't remove the term if the wdf is
    // still non-zero.
    doc.add_posting("foo", 123, 2);
    TEST_EQUAL(doc.remove_postings("foo", 1, 200), 1);
    t = doc.termlist_begin();
    TEST(t != doc.termlist_end());
    TEST(t.positionlist_begin() == t.positionlist_end());
    TEST_EQUAL(t.get_wdf(), 1);

    // Test removing the last posting is handled correctly (this case is
    // special-cased internally to be O(1)).
    t = doc.termlist_begin();
    doc.add_posting("foo", 12);
    doc.add_posting("foo", 23);
    pi = t.positionlist_begin();
    doc.remove_posting("foo", 23);
    TEST_EQUAL(*pi, 12);
    ++pi;
    TEST_EQUAL(*pi, 23);
    ++pi;
    TEST(pi == t.positionlist_end());
}

[[noreturn]]
static void
errorcopyctor_helper(Xapian::Error& error)
{
    // GCC 9 was giving a warning on the next line with -Wdeprecated-copy
    // (which is enabled by -Wextra).
    throw error;
}

/// Regression test for warning with GCC 9.
DEFINE_TESTCASE(errorcopyctor, !backend) {
    Xapian::RangeError e("test");
    try {
	errorcopyctor_helper(e);
    } catch (Xapian::Error&) {
	return;
    }
    FAIL_TEST("Expected exception to be thrown");
}

DEFINE_TESTCASE(emptydbbounds, !backend) {
    Xapian::Database db;
    TEST_EQUAL(db.get_doclength_lower_bound(), 0);
    TEST_EQUAL(db.get_doclength_upper_bound(), 0);
    // We always returned 1 here in the initial implementation.
    TEST_EQUAL(db.get_unique_terms_lower_bound(), 0);
    TEST_EQUAL(db.get_unique_terms_upper_bound(), 0);
}

// Test ESetIterator iterator_traits.
DEFINE_TESTCASE(stlesetiterator, !backend) {
    Xapian::ESet eset;
    vector<string> v;
    // This gave a compile error with stdc++ and -DGLIBCXX_DEBUG in 1.4.30:
    v.insert(v.begin(), eset.begin(), eset.end());
}

// Test MSetIterator iterator_traits.
DEFINE_TESTCASE(stlmsetiterator, !backend) {
    Xapian::MSet mset;
    vector<Xapian::docid> v;
    // In Xapian <= 1.4.30 this gave a compile error with libc++, or
    // with stdc++ and -DGLIBCXX_DEBUG:
    v.insert(v.begin(), mset.begin(), mset.end());
}

// Test PositionIterator iterator_traits.
DEFINE_TESTCASE(stlpositioniterator, !backend) {
    Xapian::Database db;
    vector<Xapian::termpos> v;
    if (db.get_doccount() > 0) {
	// In Xapian <= 1.4.30 this gave a compile error with stdc++ and
	// -DGLIBCXX_DEBUG:
	v.insert(v.begin(),
		 db.positionlist_begin(1, ""),
		 db.positionlist_end(1, ""));
    }
}

// Test PostingIterator iterator_traits.
DEFINE_TESTCASE(stlpostingiterator, !backend) {
    Xapian::Database db;
    vector<Xapian::docid> v;
    // In Xapian <= 1.4.30 this gave a compile error with stdc++ and
    // -DGLIBCXX_DEBUG:
    v.insert(v.begin(), db.postlist_begin(""), db.postlist_end(""));
}

// Test TermIterator iterator_traits.
DEFINE_TESTCASE(stltermiterator, !backend) {
    Xapian::Document doc;
    vector<string> v;
    // In Xapian <= 1.4.30 this gave a compile error with stdc++ and
    // -DGLIBCXX_DEBUG:
    v.insert(v.begin(), doc.termlist_begin(), doc.termlist_end());
}

// Test Utf8Iterator iterator_traits.
DEFINE_TESTCASE(stlutf8iterator, !backend) {
    vector<unsigned> v;
    // In Xapian <= 1.4.30 this gave a compile error with stdc++ and
    // -DGLIBCXX_DEBUG:
    v.insert(v.begin(), Xapian::Utf8Iterator(""), Xapian::Utf8Iterator());
}

// Test ValueIterator iterator_traits.
DEFINE_TESTCASE(stlvalueiterator, !backend) {
    Xapian::Document doc;
    vector<string> v;
    // In Xapian <= 1.4.30 this gave a compile error with stdc++ and
    // -DGLIBCXX_DEBUG:
    v.insert(v.begin(), doc.values_begin(), doc.values_end());
}
